const otpGenerator = require('otp-generator');
var moment = require('moment'); // require
const path = require("path");
var fs = require("fs");
const fse = require('fs-extra')
const bcrypt = require("bcrypt");
const helper = require('../middleware/_helper');

const User = require('../model/user');
const ProjectModel = require('../model/projects');
const OTPModel = require("../model/otp");
const email_templates = require('../config/email_template_ids');
const emailCtrl = require('../controllers/email.controller');
const mediaUtils = require("../config/unlinkMedia");

// Create and Save a new User
exports.create = async (req, res) => {

    // Our Add user logic starts here
    try {
        // Get user input
        const { first_name, last_name, email, phone, company_name, address_line, country, state, city, zip_code } = req.body;

        // Validate user input
        if (!(email && first_name)) {
            return res.status(400).send({ "is_error": true, "message": "All input ( Email, first & last name) is required" });
        }

        // check if user already exist
        // Validate if user exist in our database
        const oldUser = await User.findOne({ email: req.body.email.toLowerCase() });

        if (oldUser) {
            return res.status(409).send({ "is_error": true, "message": "User Already Exist." });
        }

        const tempPassword = otpGenerator.generate(7);
        console.log(tempPassword,'Temporary Password');
        //Encrypt user password
        encryptedPassword = await bcrypt.hash(tempPassword, 10);

        // Create user in our database
        const user = await User.create({
            first_name,
            last_name,
            email: email.toLowerCase(), // sanitize: convert email to lowercase
            role: req.body.role,
            password: encryptedPassword,
            phone,
            company_name,
            address_line,
            country,
            state,
            city,
            zip_code
        });

        const emailData = {
            msg: {
                to: email,
                from: process.env.SUPPORT_EMAIL, // Use the email address or domain you verified above
                template_id: email_templates.USER_ONBOARDING_EMAIL_TEMPLATE_ID,
                dynamic_template_data:{
                  "first_name": user.first_name,
                  "last_name": user.last_name,
                  "password": tempPassword
                }
              },
            userDetails: user
        }
        // check if Role is inspector then only send email
        if(req.body.role === 'INSPECTOR'){

            // send Email to User with OTP
            emailCtrl.sendEmail(emailData);
        }

        // return new user
        res.status(200).json({ is_error: false, data: user });
    } catch (err) {
        console.log(err);
        res.status(500).json({ is_error: true, data: {}, message:"Error While Creating user.Please try again."+ err});

    }
    // Our Add user logic ends here

};

// Retrieve and return all users from the database.
exports.findAll = async (req, res) => {
    try {
        const user = await User.find({is_admin: false}).select("-password -__v").sort([["updatedAt",-1]]);
        return res.send({"is_error":false , data:user})
    } catch (error) {
        console.log(error);
        return res.send({ "is_error": true, "message": "An error occured" + error });
    }
};

exports.getContractors = async (req, res) => {
    try {
        const user = await User.find({is_admin: false,role: 'CONTRACTOR'}).select("-password -__v").sort([["updatedAt",-1]]);
        console.log("users", user)
        return res.send({"is_error":false , data:user})
    } catch (error) {
        console.log(error);
        return res.send({ "is_error": true, "message": "An error occured" + error });
    }
};
exports.getInspectors = async (req, res) => {
    try {
        const user = await User.find({is_admin: false,role: 'INSPECTOR'}).select("-password -__v").sort([["updatedAt",-1]]);
        console.log("users", user)
        return res.send({"is_error":false , data:user})
    } catch (error) {
        console.log(error);
        return res.send({ "is_error": true, "message": "An error occured" + error });
    }
};
exports.getOwners = async (req, res) => {
    try {
        const user = await User.find({is_admin: false,role: 'OWNER'}).select("-password -__v").sort([["updatedAt",-1]]);
        console.log("users", user)
        return res.send({"is_error":false , data:user})
    } catch (error) {
        console.log(error);
        return res.send({ "is_error": true, "message": "An error occured" + error });
    }
};
exports.getManufacturers = async (req, res) => {
    try {
        const user = await User.find({is_admin: false,role: 'MANUFACTURER'}).select("-password -__v").sort([["updatedAt",-1]]);
        return res.send({"is_error":false , data:user})
    } catch (error) {
        console.log(error);
        return res.send({ "is_error": true, "message": "An error occured" + error });
    }
};

exports.getProjectForeman = async (req, res) => {
    try {
        const user = await User.find({is_admin: false,role: 'FOREMAN'}).select("-password -__v").sort([["updatedAt",-1]]);
        return res.send({"is_error":false , data:user})
    } catch (error) {
        console.log(error);
        return res.send({ "is_error": true, "message": "An error occured" + error });
    }
};
// Delete a userData with the specified userId in the request
exports.delete = async (req, res) => {
    
    User.findByIdAndRemove(req.params.userId)
        .then(async userData => {
            if (!userData) {
                return res.status(404).send({
                    is_error: true,
                    message: "User not found with id " + req.params.userId
                });
            }
            // Delete User's created templates
            await templateModel.deleteMany({created_by:req.params.userId});
            // Delete User's device tokens
            await deviceTokenModel.deleteMany({user_id:req.params.userId});
            // Delete all images and documents for the event
            let imageLocation = path.resolve("./public/assets/images/" + req.params.userId);
            console.log('imageLocation', imageLocation)
            fse.remove(imageLocation, err => {
                if (err) return console.error(err)
                res.send({
                    is_error: false,
                    message: "User deleted successfully!"
                });
            })
            
        }).catch(err => {
            if (err.kind === 'ObjectId' || err.name === 'NotFound') {
                return res.status(404).send({
                    is_error: true,
                    message: "User not found with id " + req.params.userId
                });
            }
            return res.status(500).send({
                is_error: true,
                message: "Could not delete User with id " + req.params.userId
            });
        });
};

exports.getProjectsByUserID = async (req, res) => {
    try {
        console.log("contractord id", req.params.userId)
        const userDetails = await User.findOne({_id:req.params.userId});
        console.log("found user", userDetails)
        let query = { is_deleted : false};
        if(userDetails && userDetails.role === 'CONTRACTOR'){
            query['assignedContractor'] = req.params.userId ;
        }
        else if(userDetails && userDetails.role === 'MANUFACTURER'){
            query['assignedManufacturers'] = req.params.userId ;
        }
        else if(userDetails && userDetails.role === 'INSPECTOR'){
            query['assignedInspector'] = req.params.userId ;
        }
        else if(userDetails && userDetails.role === 'OWNER'){
            query['assignedOwner'] = req.params.userId ;
        }
        else if(userDetails && userDetails.role === 'MANAGER'){
            query['assignedManager'] = req.params.userId ;
        }
        else {
            return res.send({"is_error":false , data:[]})
        }
        console.log("query", query)
        console.log("role", userDetails.role)
        const projects = await ProjectModel.find(query)
        .populate([{path:'assignedManufacturers' , select:['first_name', 'last_name','email']}]).sort([["updatedAt",-1]]);
        console.log("projects", projects)
        if(!projects){
            return res.status(404).send({ "is_error": true, "message": "An error occured" + error });
        }
        return res.send({"is_error":false , data:projects})
    } catch (error) {
        console.log(error);
        return res.status(404).send({ "is_error": true, "message": "An error occured" + error });
    }
};

// upload company logo for a User
exports.uploadCompanyLogo = async (req, res) => {

    console.log("req.files", req.files)
     // For Company Logo
    if (req?.files?.company_logo) {
        const imagePath = "public/uploads/company_logo/" + req.files['company_logo'][0].filename;
    
        User.findByIdAndUpdate(req.body.userId, {
            company_logo: imagePath,
        },{new:true})
        .then(userData => {
            if (!userData) {
                mediaUtils.unlink(imagePath);
                return res.status(404).send({
                    is_error: true,
                    message: "Error while updating company logo."
                });
            }
            // now removed existing company_logo image
            const dataSendToUI = {
                first_name: userData.first_name,
                last_name: userData.last_name,
                email: userData.email.toLowerCase(),
                phone: userData.phone,
                address_line: userData.address_line,
                country: userData.country,
                state: userData.state,
                city: userData.city,
                zip_code: userData.zip_code,
                role: userData.role,
                profileImage: userData.profileImage,
                company_logo: userData.company_logo
            }
            return res.send({
                is_error: false,
                data: dataSendToUI,
                message: "Company Logo has been updated successfully."
            });
        }).catch(err => {
            const imagePath = "public/uploads/company_logo/" + req.files['company_logo'][0].filename;
    
            mediaUtils.unlink(imagePath);
    
            if (err.kind === 'ObjectId') {
                return res.status(404).send({
                    is_error: true,
                    message: "Error while updating company logo."
                });
            }
            return res.status(500).send({
                is_error: true,
                message: "Error while updating company logo."
            });
        }); 
    }else{
        return res.send({
            is_error: false,
            data: [],
            message: "Operation Success. You have not made any changes."
        });
    }
};

// User specific APIS

// Update a user identified by the userId in the request
exports.updateMyAccount = async (req, res) => {

    // Validate Request
    if (!(req.body.email && req.body.first_name && req.body.last_name)) {
        return res.status(400).send({ "is_error": true, "message": "All inputs are required." });
    }
    const loggedInUser = helper.getCurrentUser(req.headers["x-access-token"]);
    if (loggedInUser && loggedInUser.data && loggedInUser.data.user_id) {
        // Validate if email already exist in our database
        const oldUser = await User.find({ email: req.body.email.toLowerCase(), _id: { $ne: loggedInUser.data.user_id } });
        console.log("old user", oldUser)
        if (oldUser.length > 0) {
            return res.status(409).send({ "is_error": true, "message": "Email not available. Please use some different Email to update your profile details." });
        }
        // Find note and update it with the request body
        await User.findByIdAndUpdate(loggedInUser.data.user_id, {
            first_name: req.body.first_name,
            last_name: req.body.last_name,
            email: req.body.email.toLowerCase(),
            phone: req.body.phone,
            address_line: req.body.address_line,
            country: req.body.country,
            state: req.body.state,
            city: req.body.city,
            zip_code: req.body.zip_code
        }, { new: true })
            .then(userData => {
                if (!userData) {
                    return res.status(404).send({
                        is_error: true,
                        message: "User details not found."
                    });
                }
                const dataSendToUI = {
                    first_name: userData.first_name,
                    last_name: userData.last_name,
                    email: userData.email.toLowerCase(),
                    phone: userData.phone,
                    address_line: userData.address_line,
                    country: userData.country,
                    state: userData.state,
                    city: userData.city,
                    zip_code: userData.zip_code,
                    role: userData.role,
                    profileImage: userData.profileImage
                }
                res.send({
                    is_error: false,
                    data: dataSendToUI,
                    message: "Profile details has been updated sucessfully."
                });
            }).catch(err => {
                if (err.kind === 'ObjectId') {
                    return res.status(404).send({
                        is_error: true,
                        message: "User details not found with provided Id."
                    });
                }
                return res.status(500).send({
                    is_error: true,
                    message: "Error updating the account details."
                });
            });
    } else {
        // Throw error
        return res.status(400).send({ "is_error": true, "message": "Unauthorized Access!" });
    }
};

// Change my Password User
exports.changeMyPassword = async (req, res) => {
    // Our logic starts here
    try {
        // Get user input
        const { current_pwd } = req.body;

        // Validate user input
        if (!(current_pwd)) {
            return res.status(400).send({ "is_error": true, "message": "Please provide current Password." });
        } else {
            const loggedInUser = helper.getCurrentUser(req.headers["x-access-token"]);
            // Get current Logged In user detail
            const user = await User.findOne({ _id: loggedInUser.data.user_id });
            console.log("found user", user);
            console.log(await bcrypt.compare(current_pwd, user.password))
            if (user && (await bcrypt.compare(current_pwd, user.password))) {
                // const user = await User.findOne({ email });

                const OTP = otpGenerator.generate(4, { digits: true, lowerCaseAlphabets: false, upperCaseAlphabets: false, specialChars: false });
                const expireTime = moment(new Date(), "DD-MM-YYYY hh:mm:ss")
                    .add(1, 'minutes')
                    .format('DD/MM/YYYY hh:mm:ss');

                const emailData = {
                    msg: {
                        to: [user.email],
                        from: process.env.SUPPORT_EMAIL, // Use the email address or domain you verified above
                        template_id: email_templates.CHANGE_PASSWORD_EMIL_TEMPLATE_ID,
                        dynamic_template_data:{
                          "subject":"Roof inspection OTP- Change Password Confirmation",
                          "first_name": user.first_name,
                          "last_name": user.last_name,
                          "OTP": OTP
                        }
                      },
                    userDetails: user
                }

                const OTP_Payload = {
                    user_id: user._id,
                    email: user.email,
                    otpvalue: OTP
                };
                console.log("OTP Payload", OTP_Payload)
                let OTP_data = await OTPModel.findOneAndUpdate({ email: user.email, user_id: user._id }, OTP_Payload, {
                    new: true,
                    upsert: true // Make this update into an upsert
                });
                if (OTP_data) {
                    // send Email to User with OTP
                      emailCtrl.sendEmail(emailData);
                    return res.send({ is_error: false, data: { }, message: "OTP has been sent to email . Please verify the OTP." });

                } else {
                    return res.status(400).send({ "is_error": true, "message": "Error while change password. Please try after some time." });
                }
            } else {
                return res.status(400).send({ "is_error": true, "message": "Invalid Password. Please provide correct password." });
            }
        }
    } catch (err) {
        console.log(err);
    }

};

// Validate otp for change password.
exports.validateOtp = async (req, res) => {
    // Our logic starts here
    try {
        // Get user input
        const { otp } = req.body;

        // Validate user input
        if (!(otp)) {
            return res.status(400).send({ "is_error": true, "message": "Please provide OTP." });
        } else {
            const loggedInUser = helper.getCurrentUser(req.headers["x-access-token"]);
            // Get current Logged In user detail
            const loggedinUserDetails = await User.findOne({ _id: loggedInUser.data.user_id });
            // Validate if otp exist in our database
            const user = await OTPModel.findOne({ 'email': loggedinUserDetails.email, 'otpvalue': req.body.otp });

            if (user && loggedinUserDetails) {

                // Find note and update it with the request body
                OTPModel.findByIdAndUpdate(user._id, { otp_verfied: true })
                    .then(otpData => {
                        if (!otpData) {
                            return res.status(404).send({
                                is_error: true,
                                message: "Invalid OTP.Either you have entered wrong OTP or your OTP has been expired."
                            });
                        }
                        return res.send({
                            is_error: false,
                            message: "OTP Verified successfully."
                        });
                    }).catch(err => {
                        if (err.kind === 'ObjectId') {
                            return res.status(404).send({
                                is_error: true,
                                message: "Invalid OTP.Either you have entered wrong OTP or your OTP has been expired."
                            });
                        }
                        return res.status(500).send({
                            is_error: true,
                            message: "Invalid OTP.Either you have entered wrong OTP or your OTP has been expired."
                        });
                    });

            } else {
                return res.status(400).send({ "is_error": true, "message": "Invalid OTP.Either you have entered wrong OTP or your OTP has been expired" });
            }
        }
    } catch (err) {
        console.log(err);
    }
    // Our logic ends here
};

// update user password
exports.updatePassword = async (req, res) => {
    // Our logic starts here
    try {
        // Get user input
        const { newpassword } = req.body;

        // Validate user input
        if (!(newpassword)) {
            return res.status(400).send({ "is_error": true, "message": "Please provide passowrd" });
        } else {
            const loggedInUser = helper.getCurrentUser(req.headers["x-access-token"]);
            // Get current Logged In user detail
            const user = await User.findOne({ _id: loggedInUser.data.user_id });
            // Validate if otp exist in our database
            const OTPfound = await OTPModel.findOne({ 'email': user.email, 'otpvalue': req.body.otp });

            if (user && OTPfound && OTPfound.otp_verfied) {
                //Encrypt user password
                encryptedPassword = await bcrypt.hash(newpassword, 10);

                // delete otp entry from DB
                OTPModel.findByIdAndRemove(OTPfound._id)
                    .then(otpData => {
                        if (!otpData) {
                            return res.status(404).send({
                                is_error: true,
                                message: "Invalid OTP ! Either you have entered wrong OTP or your OTP has been expired."
                            });
                        }
                        // Find user and update password
                        User.findByIdAndUpdate(user._id, {
                            password: encryptedPassword,
                        })
                            .then(userData => {
                                if (!userData) {
                                    return res.status(404).send({
                                        is_error: true,
                                        message: "Error while updating password."
                                    });
                                }
                                const emailData = {
                                    msg: {
                                        to: userData.email,
                                        from: process.env.SUPPORT_EMAIL, // Use the email address or domain you verified above
                                        template_id: email_templates.PSWD_RESET_BY_ADMIN_CONFIRMATION_EMAIL,
                                        dynamic_template_data:{
                                          "subject":"New Password Details- Roof Inspections Reports",
                                          "first_name": userData.first_name,
                                          "last_name": userData.last_name,
                                          "password": req.body.newpassword
                                        }
                                      },
                                    userDetails: user
                                }
                                // send Email to User with OTP
                                emailCtrl.sendEmail(emailData);
                                return res.send({
                                    is_error: false,
                                    message: "Password has been changed successfully."
                                });
                            }).catch(err => {
                                if (err.kind === 'ObjectId') {
                                    return res.status(404).send({
                                        is_error: true,
                                        message: "Error while updating Password."
                                    });
                                }
                                return res.status(500).send({
                                    is_error: true,
                                    message: "Error while updating Password."
                                });
                            });
                    }).catch(err => {
                        if (err.kind === 'ObjectId' || err.name === 'NotFound') {
                            return res.status(404).send({
                                is_error: true,
                                message: "Invalid OTP ! Either you have entered wrong OTP or your OTP has been expired."
                            });
                        }
                        return res.status(500).send({
                            is_error: true,
                            message: "Invalid OTP ! Either you have entered wrong OTP or your OTP has been expired."
                        });
                    });

            } else {
                res.status(400).send({ "is_error": true, "message": "Error while updating Password. Invalid OTP ! Either you have entered wrong OTP or your OTP has been expired." });
            }
        }
    } catch (err) {
        console.log(err);
    }
    // Our logic ends here
};

// Update a user identified by the userId in the request
exports.changeProfilePic = async (req, res) => {

    console.log("req.files", req.files)
    const imagePath = "public/uploads/profile_picture/" + req.files['profile_pic'][0].filename;
    
    const loggedInUser = helper.getCurrentUser(req.headers["x-access-token"]);
    User.findByIdAndUpdate(loggedInUser.data.user_id, {
        profileImage: imagePath,
    })
    .then(userData => {
        if (!userData) {
            return res.status(404).send({
                is_error: true,
                message: "Error while updating profile image."
            });
        }
        const dataSendToUI = {
            first_name: userData.first_name,
            last_name: userData.last_name,
            email: userData.email.toLowerCase(),
            phone: userData.phone,
            address_line: userData.address_line,
            country: userData.country,
            state: userData.state,
            city: userData.city,
            zip_code: userData.zip_code,
            role: userData.role,
            profileImage: userData.profileImage
        }
        return res.send({
            is_error: false,
            data: dataSendToUI,
            message: "Profile Image has been updated successfully."
        });
    }).catch(err => {
        if (err.kind === 'ObjectId') {
            return res.status(404).send({
                is_error: true,
                message: "Error while updating profile image."
            });
        }
        return res.status(500).send({
            is_error: true,
            message: "Error while updating profile image."
        });
    });
};

// Change user status
exports.changeStatus = async (req, res) => {
    await User.findByIdAndUpdate(req.params.userId, {
        is_active: req.body.is_active
    }, { new: true })
        .then(userData => {
            if (!userData) {
                return res.status(404).send({
                    is_error: true,
                    message: "Invalid User Details."
                });
            }
            const notifyMsg = req.body.is_active ? 'Activated' : 'In-activated';
            res.send({'message': 'User Profile has been ' +' '+ notifyMsg });
        }).catch(err => {
            if (err.kind === 'ObjectId') {
                return res.status(404).send({
                    is_error: true,
                    message: "User details not found with provided Id."
                });
            }
            return res.status(500).send({
                is_error: true,
                message: "Error updating the user status."
            });
        });
};

// logged in Admin user details- Admin user
exports.myDetails = (req, res) => {
    const loggedInUser = helper.getCurrentUser(req.headers["x-access-token"]);
    User.findById(loggedInUser.data.user_id).select("-password -__v")
        .then(userData => {
            if (!userData) {
                return res.status(404).send({
                    is_error: true,
                    message: "User not found with id " + loggedInUser.data.user_id
                });
            }
            res.send({ is_error: false, data: userData });
        }).catch(err => {
            if (err.kind === 'ObjectId') {
                return res.status(404).send({
                    is_error: true,
                    message: "User not found with id " + loggedInUser.data.user_id
                });
            }
            return res.status(500).send({
                is_error: true,
                message: "Error retrieving user with id " + loggedInUser.data.user_id
            });
        });
};

// send Contact Us query
exports.changeAdminPassword = async (req, res) => {
    // Our logic starts here
    try {
        // Get user input
        const { currentPwd, newPwd } = req.body;

        // Validate user input
        if (!(currentPwd && newPwd)) {
            return res.status(400).send({ "is_error": true, "message": "Please provide current Password." });
        } else {
            const loggedInUser = helper.getCurrentUser(req.headers["x-access-token"]);
            // Get current Logged In user detail
            const user = await User.findOne({ _id: loggedInUser.data.user_id });
            
            if (user && (await bcrypt.compare(currentPwd, user.password))) {
                //Encrypt user password
                encryptedPassword = await bcrypt.hash(newPwd, 10);
                // Find user and update password
                User.findByIdAndUpdate(user._id, {
                    password: encryptedPassword,
                })
                    .then(userData => {
                        if (!userData) {
                            return res.status(404).send({
                                is_error: true,
                                message: "Error while updating password."
                            });
                        }
                        return res.send({
                            is_error: false,
                            message: "Password changed successfully."
                        });
                    }).catch(err => {
                        if (err.kind === 'ObjectId') {
                            return res.status(404).send({
                                is_error: true,
                                message: "Error while updating Password."
                            });
                        }
                        return res.status(500).send({
                            is_error: true,
                            message: "Error while updating Password."
                        });
                    });
            } else {
                return res.status(400).send({ "is_error": true, "message": "Invalid Password. Please provide your correct current password." });
            }
        }
    } catch (err) {
        console.log(err);
    }

};


// send Contact Us query
exports.changeUserPasswordByUserIDViaAdmin = async (req, res) => {
    // Our logic starts here
    try {
        // Get user input
        const { userId } = req.body;

        // Validate user input
        if (!req.body.newpassword) {
            return res.status(400).send({ "is_error": true, "message": "Please provide new password details." });
        } 
        if (!(userId)) {
            return res.status(400).send({ "is_error": true, "message": "Please select correct user for reset password." });
        } else {
            const loggedInUser = helper.getCurrentUser(req.headers["x-access-token"]);
            if(!loggedInUser || !loggedInUser.data){
                return res.status(400).send({ "is_error": true, "message": "You are not authorized to perform this operation." });
            }
            if(loggedInUser.data.role !='ADMIN'){
                return res.status(400).send({ "is_error": true, "message": "You are not authorized to perform this operation." });
            }
            // Get current Logged In user detail
           User.findOne({ _id: req.body.userId })
            .then(async user => {
                if (!user) {
                    return res.status(404).send({
                        is_error: true,
                        message: "User Details not found."
                    });
                }
                console.log("found user", user)
                if (user) {
                console.log("found user ndise", user)

                    const tempPassword = otpGenerator.generate(7);
                    //Encrypt user password
                    encryptedPassword = await bcrypt.hash(req.body.newpassword, 10);
                    // Find user and update password
                    User.findByIdAndUpdate(user._id, {
                        password: encryptedPassword,
                    })
                    .then(userData => {
                        if (!userData) {
                            return res.status(404).send({
                                is_error: true,
                                message: "Error while updating user's password."
                            });
                        }
                        const emailData = {
                            msg: {
                                to: userData.email,
                                from: process.env.SUPPORT_EMAIL, // Use the email address or domain you verified above
                                template_id: email_templates.PSWD_RESET_BY_ADMIN_CONFIRMATION_EMAIL,
                                dynamic_template_data:{
                                  "subject":"New Password Details- Roof Inspections Reports",
                                  "first_name": userData.first_name,
                                  "last_name": userData.last_name,
                                  "password": req.body.newpassword
                                }
                              },
                            userDetails: user
                        }
                        // send Email to User with OTP
                        emailCtrl.sendEmail(emailData);
                        return res.send({
                            is_error: false,
                            message: "User Password has been changed successfully. An email has been sent to user with new password details."
                        });
                    }).catch(err => {
                        console.log("error", err)
                        if (err.kind === 'ObjectId') {
                            return res.status(404).send({
                                is_error: true,
                                message: "Error while updating user's Password."
                            });
                        }
                        return res.status(500).send({
                            is_error: true,
                            message: "Error while updating user's Password."
                        });
                    });
                } else {
                    return res.status(400).send({ "is_error": true, "message": "Error Occured. Please try after some time." });
                }
            }).catch(err => {
                console.log("err", err)
                if (err.kind === 'ObjectId') {
                    return res.status(404).send({
                        is_error: true,
                        message: "User Details not found.Invalid user ref id "
                    });
                }
                return res.status(500).send({
                    is_error: true,
                    message: "Error retrieving user details"
                });
            });
        }
    } catch (err) {
        console.log(err);
    }

};

// Find a single user with a email
exports.findUserByEmail = async (req, res) => {
    console.log("req.body", req.body)
    // Validate user input
    if (!req.body.email) {
        return res.status(400).send({ "is_error": true, "message": "Please provide user email to search." });
    }
    const loggedInUser = helper.getCurrentUser(req.headers["x-access-token"]);
    const query = {$or:[  {'first_name':{ $regex: '.*' + req.body.email+ '.*', $options: 'i' }},
            {'last_name':{ $regex: '.*' + req.body.email + '.*', $options: 'i' }},
            {'email': req.body.email}], 
            is_admin: false, is_active: true, _id: { $ne: loggedInUser.data.user_id } }
        // await User.find({is_admin: false, is_active: true, email: req.body.email ,  _id: { $ne: loggedInUser.data.user_id }}).select("-password -__v -events -templates -createdAt -updatedAt -role -is_active -is_admin")
        await User.find(query).select("-password -__v -events -templates -createdAt -updatedAt -role -is_active -is_admin")
        .then(userData => {
            if (!userData) {
                return res.status(404).send({
                    is_error: true,
                    message: "No user found for :  " + req.body.email
                });
            }
            res.send({ is_error: false, data: userData});
        }).catch(err => {
            if (err.kind === 'ObjectId') {
                return res.status(404).send({
                    is_error: true,
                    message: "No user found for : " + req.body.email
                });
            }
            return res.status(500).send({
                is_error: true,
                message: "Error in searching user."
            });
        });
};

processImageUploadData = async (req,res, imagePath , dbData) =>{
    const loggedInUser = helper.getCurrentUser(req.headers["x-access-token"]);
   // Find note and update it with the request body
   User.findByIdAndUpdate(loggedInUser.data.user_id, {
    profileImage: imagePath,
    }, { new: true })
        .then(userData => {
            if (!userData) {
                return res.status(404).send({
                    is_error: true,
                    message: "User details not found."
                });
            }
            return res.send({
                is_error: false,
                data: userData,
                message: "Profile Picture has been updated successfully."
            });
        }).catch(err => {
            dbData.forEach(item => {
                if (fs.existsSync(item)) {
                    fs.unlink(item, (err) => {
                        if (err) {
                            console.log(err);
                        }
                        console.log('deleted');
                    })
                }
            })
            if (err.kind === 'ObjectId') {
                return res.status(404).send({
                    is_error: true,
                    message: "User details not found with provided Id."
                });
            }
            return res.status(500).send({
                is_error: true,
                message: "Error updating the account details."
            });
        });
}


  exports.testing = async(req, res)=>{

    console.log(process.env.STRIPE_KEY);
    process.env.STRIPE_KEY = "sdgddgdgddgddgdgdgddg";
    console.log(process.env.STRIPE_KEY);

    exec("rm -rf /c/Users/lenovo/Documents/reportingTool/frontEnd/dist", (error, stdout, stderr) => {
        if (error) {
            console.log(`error: ${error.message}`);
            return res.send(error.message);
        }
        if (stderr) {
            console.log(`stderr: ${stderr}`);
            return res.send(stderr);
        }
        console.log(`stdout: ${stdout}`);
        res.send("done")
    });
    // exec("forever start server.js", (error, stdout, stderr) => {
    //     if (error) {
    //         console.log(`error: ${error.message}`);
    //         return res.send(error.message);
    //     }
    //     if (stderr) {
    //         console.log(`stderr: ${stderr}`);
    //         return res.send(stderr);
    //     }
    //     console.log(`stdout: ${stdout}`);
    //     res.send("done")
    // });
  }

  const { exec } = require("child_process");

  exports.testEmail = async(req, res)=> {
    const emailData = {
        msg: {
            to: ['shivam201065@gmail.com'],
            from: process.env.SUPPORT_EMAIL, // Use the email address or domain you verified above
            template_id: email_templates.CHANGE_PASSWORD_EMIL_TEMPLATE_ID,
            dynamic_template_data:{
              "subject":"Roof inspection OTP- Change Password Confirmation",
              "first_name": "Shivam",
              "last_name": 'S',
              "OTP": '12345678'
            },
            // replyTo: {
            //     email: 'devworker20@gmail.com',
            //     name: 'Example Customer Service Team'
            //   },
          }
    }

    // send Email to User with OTP
    emailCtrl.sendEmail(emailData);
  }


//   RPC Manager Functions
exports.getRpcManagers = async (req, res) => {
    try {
        const user = await User.find({is_admin: false,role: 'RPC_MANAGER'}).select("-password -__v").sort([["updatedAt",-1]]);
        
        return res.send({"is_error":false , data:user})
    } catch (error) {
        console.log(error);
        return res.send({ "is_error": true, "message": "An error occured" + error });
    }
};

// Create New RPC Manager
exports.addRpcManager = async (req, res) => {

    // Our Add user logic starts here
    try {
        // Get user input
        const { first_name, last_name, email, phone, company_name, address_line, country, state, city, zip_code } = req.body;

        // Validate user input
        if (!(email && first_name)) {
            return res.status(400).send({ "is_error": true, "message": "All input ( Email, first & last name) is required" });
        }

        // check if user already exist
        // Validate if user exist in our database
        const oldUser = await User.findOne({ email: req.body.email.toLowerCase() });

        if (oldUser) {
            return res.status(409).send({ "is_error": true, "message": "User Already Exist." });
        }

        const tempPassword = otpGenerator.generate(7);
        console.log(tempPassword,'Temporary Password');
        //Encrypt user password
        encryptedPassword = await bcrypt.hash(tempPassword, 10);

        // Create user in our database
        const user = await User.create({
            first_name,
            last_name,
            email: email.toLowerCase(), // sanitize: convert email to lowercase
            role: 'RPC_MANAGER',
            password: encryptedPassword,
            phone,
            company_name,
            address_line,
            country,
            state,
            city,
            zip_code
        });

        const emailData = {
            msg: {
                to: email,
                from: process.env.SUPPORT_EMAIL, // Use the email address or domain you verified above
                template_id: email_templates.USER_ONBOARDING_EMAIL_TEMPLATE_ID,
                dynamic_template_data:{
                  "first_name": user.first_name,
                  "last_name": user.last_name,
                  "password": tempPassword
                }
              },
            userDetails: user
        }
        // check if Role is inspector then only send email
        if(req.body.role === 'INSPECTOR'){

            // send Email to User with OTP
            emailCtrl.sendEmail(emailData);
        }

        // return new user
        res.status(200).json({ is_error: false, data: user });
    } catch (err) {
        console.log(err);
        res.status(500).json({ is_error: true, data: {}, message:"Error While Creating user.Please try again."+ err});

    }
    // Our Add user logic ends here

};

// Find a single user with a userId
exports.findOne = async (req, res) => {
    User.findById(req.params.userId).select("-password -__v")
        // .populate({
        //     path:'assigned_projects'
        // })
        .then(userData => {
            if (!userData) {
                return res.status(404).send({
                    is_error: true,
                    message: "User not found with id " + req.params.userId
                });
            }
            res.send({ is_error: false, data: userData });
        }).catch(err => {
            console.log("err", err)
            if (err.kind === 'ObjectId') {
                return res.status(404).send({
                    is_error: true,
                    message: "User not found with id " + req.params.userId
                });
            }
            return res.status(500).send({
                is_error: true,
                message: "Error retrieving user with id " + req.params.userId
            });
        });
};

// Update a user identified by the userId in the request
exports.update = async (req, res) => {
    console.log("req.body", req.body);
    console.log("req.params", req.params)
    // Validate Request
    if (!(req.body.email && req.body.first_name)) {
        return res.status(400).send({ "is_error": true, "message": "All inputs are required." });
    }
    // check if user already exist
    // Validate if user exist in our database
    const oldUser = await User.find({ email: req.body.email.toLowerCase(), _id: { $ne: req.params.userId } });
    console.log("old user", oldUser)
    if (oldUser.length > 0) {
        return res.status(409).send({ "is_error": true, "message": "User already exist with this email" });
    }
    // Find user and update it with the request body
    await User.findByIdAndUpdate(req.params.userId, {
        first_name: req.body.first_name,
        last_name: req.body.last_name,
        email: req.body.email.toLowerCase(), //sanitize convert to lowercase
        phone: req.body.phone,
        company_name: req.body.company_name,
        address_line: req.body.address_line,
        country: req.body.country,
        state: req.body.state,
        city: req.body.city,
        zip_code: req.body.zip_code
    }, { new: true })
        .then(userData => {
            if (!userData) {
                return res.status(404).send({
                    is_error: true,
                    message: "User details not found."
                });
            }
            res.send({
                is_error: false,
                data: userData
            });
        }).catch(err => {
            if (err.kind === 'ObjectId') {
                return res.status(404).send({
                    is_error: true,
                    message: "User details not found with provided Id."
                });
            }
            return res.status(500).send({
                is_error: true,
                message: "Error updating the account details."
            });
        });
};