/**
 * Removes extension from file
 * @param {string} file - filename
 */
exports.removeExtensionFromFile = file => {
  return file
    .split(".")
    .slice(0, -1)
    .join(".")
    .toString();
};


/**
 * Removes extension from file
 * @param {string} file - filename
 */
exports.unsetFields = param => {
  Object.keys(param).forEach((k) => {
    if (!param[k] && typeof param[k] !== 'boolean') {
      param.$unset = { ...param?.$unset, [k]: '' }
      delete param[k]
    }
  });
  return param
};

/**
 * Gets IP from user
 * @param {*} req - request object
 */
exports.getIP = req => req.headers['x-forwarded-for']?.split(',').shift() || req.socket?.remoteAddress

/**
 * Gets browser info from user
 * @param {*} req - request object
 */
exports.getBrowserInfo = req => req.headers["user-agent"];


/**
 * Gets browser info from user
 * @param {*} req - request object
 */
exports.getRandomColors = () => {
  let colors = ['success', 'danger', 'warning', 'info', 'secondary', "primary"]
  return colors[Math.floor(Math.random() * colors.length)]
}
/**
 * Gets country from user using CloudFlare header 'cf-ipcountry'
 * @param {*} req - request object
 */
exports.getCountry = req =>
  req.headers["cf-ipcountry"] ? req.headers["cf-ipcountry"] : "XX";

/**
 * Handles error by printing to console in development env and builds and sends an error response
 * @param {Object} res - response object
 * @param {Object} err - error object
 */
exports.handleError = (res, err) => {
  // Prints error in console
  if (process.env.NODE_ENV === "development") {
    console.log(err);
  }
  // Sends error to user
  res.status(err.code).json({
    errors: {
      msg: err.message
    }
  });
};

/**
 * Builds error object
 * @param {number} code - error code
 * @param {string} message - error text
 */
exports.buildErrObject = (code, message) => {
  return {
    code,
    message
  };
};


/**
 * Builds success object
 * @param {string} message - success text
 */
exports.buildSuccObject = message => {
  return {
    msg: message
  };
};

/**
 * Checks if given ID is good for MongoDB
 * @param {string} id - id to check
 */
exports.isIDGood = async id => {
  return new Promise((resolve, reject) => {
    const goodID = String(id).match(/^[0-9a-fA-F]{24}$/);
    return goodID
      ? resolve(id)
      : reject(exports.buildErrObject(422, "ID_MALFORMED"));
  });
};

/**
 * Item not found
 * @param {Object} err - error object
 * @param {Object} item - item result object
 * @param {Object} reject - reject object
 * @param {string} message - message
 */
exports.itemNotFound = (err, item, reject, message) => {
  if (err) {
    reject(exports.buildErrObject(422, err.message));
  }
  if (!item) {
    reject(exports.buildErrObject(404, message));
  }
};

/**
 * Item already exists
 * @param {Object} err - error object
 * @param {Object} item - item result object
 * @param {Object} reject - reject object
 * @param {string} message - message
 */
exports.itemAlreadyExists = (err, item, reject, message) => {
  if (err) {
    reject(exports.buildErrObject(422, err.message));
  }
  if (item) {
    reject(exports.buildErrObject(422, message));
  }
};

/**

* Removes empty fields from object
* @param {Object} param - payload
*/
exports.removeEmptyKeys = (param) => {
  Object.keys(param).forEach((k) => {
    if (!param[k] && typeof param[k] !== "boolean")
      delete param[k]
  });
  return param
};