const {
  buildSuccObject,
  buildErrObject,
  itemNotFound,
  itemAlreadyExists
} = require("../services/utils");


/**
 * Builds sorting
 * @param {string} sort - field to sort from
 * @param {number} order - order for query (1,-1)
 */
const buildSort = (sort, order) => {
  const sortBy = {};
  sortBy[sort] = order;
  return sortBy;
};

/**
 * Hack for mongoose-paginate, removes 'id' from results
 * @param {Object} result - result object
 */
const cleanPaginationID = (result) => {
  result?.docs?.map((element) => delete element.id);
  return result;
};

/**
 * Builds initial options for query
 * @param {Object} query - query object
 */
const listInitOptions = async (req) => {
  return new Promise((resolve) => {
    const order = req.query.order || -1;
    const sort = req.query.sort || "createdAt";
    const sortBy = buildSort(sort, order);
    const page = parseInt(req.query.page, 10) || 1;
    const limit = parseInt(req.query.limit, 10) || 10;
    const options = {
      sort: sortBy,
      lean: true,
      page,
      limit,
    };
    if (req.populate) options.populate = req.populate;
    resolve(options);
  });
};


module.exports = {
  /**
   * Checks the query string for filtering records
   * query.filter should be the text to search (string)
   * query.fields should be the fields to search into (array)
   * @param {Object} query - query object
   */
  async checkQueryString(query) {
    return new Promise((resolve, reject) => {
      try {
        let data = { deleted: false }
        if (query.extraFilter) {
          data = {
            ...data,
            $or: [
              { status: query.extraFilter }
            ]
          }
        }
        if (query?.filter && query?.fields) {
          const array = [];
          // Takes fields param and builds an array by splitting with ','
          const arrayFields = query.fields.split(",");
          // Adds SQL Like %word% with regex
          arrayFields.map(item => {
            array.push({
              [item]: {
                $regex: new RegExp(`.*${query.filter.trim()}`, "i")
              }
            });
          });
          // Puts array result in data
          data.$or = array;
        }
        resolve(data);
      } catch (err) {
        console.log(err.message);
        reject(buildErrObject(422, "ERROR_WITH_FILTER"));
      }
    });
  },


  /**
   * Gets items from database
   * @param {Object} req - request object
   * @param {Object} query - query object
   */
  async getItems(req, model, query = {}) {
    const options = await listInitOptions(req);
    return new Promise((resolve, reject) => {
      model.paginate(query, options, (err, items) => {
        if (err) {
          reject(buildErrObject(422, err.message));
        }
        resolve(cleanPaginationID(items));
      });
    });
  },

  /**
   * Gets item from database by id
   * @param {string} id - item id
   */
  async getItem(id, model) {
    return new Promise((resolve, reject) => {
      model.findById(id, (err, item) => {
        itemNotFound(err, item, reject, "NOT_FOUND");
        resolve(item);
      });
    });
  },

  /**
   * Gets item from database by id
   * @param {string} id - item id
   */
  async getItemWithPopulate(id, model, option = []) {
    return new Promise((resolve, reject) => {
      model
        .findById(id)
        .populate(option)
        .exec((err, res) =>
          err ? itemNotFound(err, item, reject, "NOT_FOUND") : resolve(res)
        );
    });
  },

  /**
   * Creates a new item in database
   * @param {Object} req - request object
   */
  async createItem(req, model) {
    return new Promise((resolve, reject) => {
      model.create(req, (err, item) => {
        if (err) {
          reject(buildErrObject(422, err.message));
        }
        resolve(item);
      });
    });
  },

  /**
   * Updates an item in database by id
   * @param {string} id - item id
   * @param {Object} req - request object
   */
  async updateItem(id, model, req, option = []) {
    return new Promise((resolve, reject) => {
      model
        .findByIdAndUpdate(id, req, {
          new: true,
          runValidators: true,
        })
        .populate(option)
        .exec((err, res) =>
          err ? itemNotFound(err, req, reject, "NOT_FOUND") : resolve(res)
        );
    });
  },

  /**
 * Updates an item in database by id
 * @param {string} param - object
 * @param {Object} req - request object
 */
  async updateAndCreateItem(param, model, req) {
    return new Promise((resolve, reject) => {
      model
        .findOneAndUpdate(param, req, {
          new: true,
          upsert: true
        })
        .exec((err, res) =>
          err ? itemNotFound(err, req, reject, "NOT_FOUND") : resolve(res)
        );
    });
  },

  /**
    * Updates an multi item in database by ids
    * @param {Aray} id - item ids
    * @param {Object} req - request object
    */
  async updateMultiItem(ids, model, req) {
    return new Promise((resolve, reject) => {
      model.updateMany({ _id: { $in: ids } },
        { $set: req },
        { multi: true }
      ).exec((err, res) =>
        err ? itemNotFound(err, item, reject, "NOT_FOUND") : resolve(res)
      );
    });
  },

  /**
   * Soft delete an multi/single item in database by ids
   * @param {Aray/String} ids - item ids
   */
  async softDeleteItems(id, model) {
    return new Promise((resolve, reject) => {
      model.delete({ _id: id }).exec((err, res) => err ? itemNotFound(err, item, reject, "NOT_FOUND") : resolve(res));
    });
  },

  /**
   * Deletes an item from database by id
   * @param {string} id - id of item
   */
  async deleteItem(id, model) {
    return new Promise((resolve, reject) => {
      model.findByIdAndRemove(id, (err, item) => {
        itemNotFound(err, item, reject, "NOT_FOUND");
        resolve(buildSuccObject("DELETED"));
      });
    });
  },

  /**
   * Gets item from database by id
   * @param {object} params - item id
   */
  async getItemsByParams(model, params = {}, option = [], sort = {}, select = "") {
    return new Promise((resolve, reject) => {
      model
        .find({ ...params, deleted: false })
        .sort(sort)
        .select(select)
        .populate(option)
        .exec((err, res) =>
          err ? itemNotFound(err, item, reject, "NOT_FOUND") : resolve(res)
        );
    });
  },
  /**
   * Gets item from database by id
   * @param {object} params - item id
   */
  async getItemByParams(model, params, option = []) {
    params = { ...params, deleted: false }
    return new Promise((resolve, reject) => {
      model
        .findOne(params)
        .populate(option)
        .exec((err, res) =>
          err ? itemNotFound(err, res, reject, "NOT_FOUND") : resolve(res)
        );
    });
  },
  /**
  * Insert item to database
  * @param {string} id - item id
  */
  async insertItem(model, obj) {
    return new Promise((resolve, reject) => {
      new model(obj).save((err, item) => {
        itemNotFound(err, item, reject, "NOT_FOUND");
        resolve(item);
      });
    });
  },

  /**
  * Gets item from database by id
  * @param {object} params - item id
  */
  async existItemByParams(model, params) {
    return new Promise((resolve, reject) => {
      model.findOne({ ...params, deleted: false },
        (err, item) => {
          itemAlreadyExists(err, item, reject, "RECORD.ALREADY.EXISTS");
          resolve(false);
        }
      );
    });
  },

  /**
    * Gets item from database by param if associated
    * @param {object} params - item id
    */
  async isAssociated(model, params) {
    return new Promise((resolve, reject) => {
      model.findOne({ ...params, deleted: false },
        (err, item) => {
          itemAlreadyExists(err, item, reject, "RECORD.ASSOCIATTED");
          resolve(false);
        }
      );
    });
  }
};
