/*SHARED CONTROLLER : Controller will Handle all common operations between users Like 
Get profile , 
Update Profile , Change Password , Etc*/

const response = require("../../constants/response");
const MESSAGE = require("../../constants/message");
const EMAIL_TEMPLATES = require("../../constants/email_templates_id");

const { User, Notifications, UserPermission } = require("../../models");
const db = require("../../services/db")
const { getLoggedInUser } = require("../../utils/jwtToken");
const bcrypt = require("bcryptjs");
const mail = require("../../utils/notify");
const mediaUtils = require("../../utils/unlinkMedia");

var fs = require("fs");
const path = require("path");

/**
* Get owner Profile details from database by id
* @param {string} id - user id
*/
const getProfileDetails = async (req, res) => {
    const authHeader = req.headers.authorization;
    let token = authHeader.split("Bearer ")[1];
    const loggedInuser = await getLoggedInUser(token);
    const owner_id = loggedInuser.payload._id;
    try {
        const owner = await db.getItemByParams(User, { _id: owner_id });
        return response.successResponse(res, 200, owner);
    } catch (e) {
        return response.somethingErrorMsgResponse(res, 500, e);
    }
}


/**
 * Updates user Profile in database by id
 * @param {string} id - item id
 * @param {Object} req - request object
 */
const updateProfile = async (req, res) => {
    try {
        const oldUserDetails = await db.getItemByParams(User, { _id: req.body._id });

        // For profile pic
        if (req?.files?.profile_pic) {
            req.body.profile_pic = {
                path: "public/uploads/profile_picture/" + req.files.profile_pic[0].filename,
                filename: req.files.profile_pic[0].filename,
                type: "images"
            };
        }

        let results = await db.updateItem(req.body._id, User, req.body);

        if (results) {
            // Now remove previously uploaded profile image
            if (req?.files?.profile_pic) { mediaUtils.unlink(oldUserDetails.profile_pic.path); }// remove upload image(s) if any failure
            return response.successResponse(res, 200, results, MESSAGE.USER_PROFILE_UPDATE_SUCCESS)
        } else {
            mediaUtils.unlink(req.body.profile_pic.path); // remove upload image(s) if any failure
            return response.badRequest(res, 400, MESSAGE.USER_NOT_FOUND);
        }
    } catch (e) {
        console.log(e)
        mediaUtils.unlink(req.body.profile_pic.path);        // remove upload image(s) if any failure
        return response.errorMessageResponse(res, 500, e);
    }
}

const changePassword = async (req, res) => {
    // Our logic starts here
    try {
        // Get user input
        const { currentPwd, newPwd } = req.body;
        // Validate user input
        if (!currentPwd) {
            return response.badRequest(res, 400, "Please provide current Password.");
        }
        if (!newPwd) {
            return response.badRequest(res, 400, "Please provide new Password.");
        }

        const authHeader = req.headers.authorization;
        let token = authHeader.split("Bearer ")[1];
        const loggedInuser = await getLoggedInUser(token);

        // Get current Logged In user detail
        const user = await User.findOne({ _id: loggedInuser.payload._id });
        if (user && (await bcrypt.compare(currentPwd, user.password))) {
            //Encrypt user password
            encryptedPassword = await bcrypt.hashSync(newPwd, 16);
            // Find user and update password
            let results = await db.updateItem(user._id, User, { password: encryptedPassword, password_change_required: false })

            if (results) {
                // let subject = "Change Password Confirmation";
                // let text = `Hii ${results.first_name} <br> Your Password has been changed successfully. You can now login with your new Password.`;
                // mail.sendMail(results.email, subject, text);

                const emailData = {
                    msg: {
                      to: results.email,
                      from: 'Winston@roofingprojects.com', // Use the email address or domain you verified above
                      template_id: EMAIL_TEMPLATES.USER_ONBOARDING_EMAIL_TEMPLATE_ID,
                      dynamic_template_data:{
                        "first_name": results.first_name,
                        "last_name": results.last_name
                      }
                    }
                  };
                  mail.sendMailWithTemplate(emailData);
                return response.successResponse(res, 200, results, MESSAGE.CHANGE_PWD_SUCCESS);
            } else {
                return response.badRequest(res, 400, MESSAGE.PASSWORD_UPDATE_FAILURE);
            }

        } else {
            return response.badRequest(res, 400, MESSAGE.CURRENT_PWD_NOT_MATCHED);
        }

    } catch (e) {
        console.log(e)
        return response.errorMessageResponse(res, 500, e);
    }

}

/**
* Get user's specific notifications
* @param {string} id - user id
*/
const getAllNotifications = async (req, res) => {
    const authHeader = req.headers.authorization;
    let token = authHeader.split("Bearer ")[1];
    const loggedInuser = await getLoggedInUser(token);
    const user_id = loggedInuser.payload._id;
    try {
        const notification = await db.getItems(req, Notifications, {});
        return response.successResponse(res, 200, notification);
    } catch (e) {
        return response.somethingErrorMsgResponse(res, 500, e);
    }
}

// Add new Notification- temporary Method for testing
const addNotifications = async (req, res) => {
    // Validate request
    if (!req.body.title ) {
        return response.errorMessageResponse(res, 500, "Notification Title is required");
      }
    let results = await db.insertItem(Notifications, req.body);
    if(results){
        return response.successResponse(res, 200, results, "notification craeted !");
    }else{
        return response.badRequest(res, 400, "error in notification creation , Please try again.")
    }
};

// Retrieve and return all my Unread Notifications from the database.
const getMyUnreadNotifications = async (req, res) => {
try {
    const authHeader = req.headers.authorization;
    let token = authHeader.split("Bearer ")[1];
    const loggedInuser = await getLoggedInUser(token);
    
    if (req.query.limit) {
        const limitCount = parseInt(req.query.limit);
        const notificationsList = await Notifications.find( {'to_user_ids':loggedInuser.payload._id, 'readby_user_ids':loggedInuser.payload._id}).sort([["updatedAt",-1]]).limit(limitCount);
        res.send({ is_error: false, data: notificationsList, unreadCount: notificationsList.length});
    }
    else {
        const notificationsList = await Notifications.find( {'to_user_ids':loggedInuser.payload._id, 'readby_user_ids':loggedInuser.payload._id}).sort([["updatedAt",-1]]);
        res.send({is_error : false, data: notificationsList, unreadCount: notificationsList.length });
    }
} catch (error) {
    console.log(error);
    return res.send({is_error : true, message:"An error occured : " + error});
}
};

// Retrieve and return all my Notifications from the database.
const getMyNotifications = async (req, res) => {
    const authHeader = req.headers.authorization;
    let token = authHeader.split("Bearer ")[1];
    const loggedInuser = await getLoggedInUser(token);
    const user_id = loggedInuser.payload._id;
    try {
        const notification = await db.getItems(req, Notifications, { to_user_ids: user_id });
        return response.successResponse(res, 200, notification);
    } catch (e) {
        return response.somethingErrorMsgResponse(res, 500, e);
    }
};

  // Change Notification status ( Read/ Unread)
  const markNotificationAsRead = async (req, res) => {
    if (!req.body.notificationIds || req.body.notificationIds.length == 0) {
        return res.status(400).send("Please select any notification(s) for marking as read.");
      }
      const authHeader = req.headers.authorization;
      let token = authHeader.split("Bearer ")[1];
      const loggedInuser = await getLoggedInUser(token);
    
    await Notifications.updateMany(
        { _id: { $in: req.body.notificationIds }, to_user_ids : loggedInuser.payload._id },
        { $push: { readby_user_ids: loggedInuser.payload._id } },
        {multi: true}
     ).then(affectedRecords => {
        console.log('affeted records', affectedRecords)
        var msg = affectedRecords.modifiedCount + ' ' + 'Notification(s) marked as read.';
        res.send({'message': msg});
    }).catch(err => {
        return res.status(500).send({
            is_error: true,
            message: "Error while updating notifications status. Please try again."
        });
    });
};

const getuserPermission = async (req, res) => {
    try {
        console.log("req.body", req.body)
        const userPermissionsData = await UserPermission.findOne({user_id:req.body._id})
        .populate([{path:'module'}]);
        return response.successResponse(res, 200, userPermissionsData);

    } catch (e) {
        console.log(e)
        return response.somethingErrorMsgResponse(res, 500, e);

    }
}

module.exports = {
    getProfileDetails,
    updateProfile,
    changePassword,
    getAllNotifications,
    addNotifications,
    getMyUnreadNotifications,
    getMyNotifications,
    markNotificationAsRead,
    getuserPermission
}