const response = require("../../constants/response");
const MESSAGE = require("../../constants/message");
const EMAIL_TEMPLATES = require("../../constants/email_templates_id");

const { User, UserPermission } = require("../../models");
const { _value } = require("../../utils/getRandom");
const { emailStatus } = require("../../utils/getUser");
const { RPC_MANAGER } = require("../../constants/roles");
const bcrypt = require("bcryptjs");
const mail = require("../../../MEAN-Roofing/utils/notify");
const db = require("../../services/db");
const mediaUtils = require("../../utils/unlinkMedia");

/**
 * Gets All RPC Mangers from database by role
 * @param {string} role - user role
 */
const getAll_RPC = async (req, res) => {
  try {
    let query = await db.checkQueryString(req.query)
    query.role = "RPC_MANAGER"
    if(req?.query?.filter && !isNaN(req?.query?.filter))
    query.$or.push({mobile_no:Number(req.query.filter)})
    return response.successResponse(res, 200, await db.getItems(req, User, query), MESSAGE.RPC_VIEW_DATA);
  } catch (error) {
    return response.somethingErrorMsgResponse(res, 500, error);
  }
};

/**
 * Creates a new RPC Manager in database
 * @param {Object} req - request object
 */
const addRPC = async (req, res) => {
  const { first_name, last_name, email, mobile_no, city, state, country, zip_code, company_name, status, address1, address2 } = req.body;
  // For Company Logo
  if (req?.files?.company_logo) {
    req.body.company_logo = "public/uploads/" + req.files.company_logo[0].filename;
  }
  // let MailExist = await emailStatus(email);
  // // if mail exist then return message
  // if (MailExist) {
  //   if (req?.files?.company_logo) {
  //     mediaUtils.unlink(req.body.company_logo); // remove uploaded compony logo on error
  //   }
  //   return response.somethingErrorMsgResponse(res, 403, MESSAGE.EMAIL_ALREADY_REGISTERED);
  // }
  const MailExist = await User.findOne({ email: email.toLowerCase() });

  if (MailExist) {
    if (req?.files?.company_logo) {
      mediaUtils.unlink(req.body.company_logo); // remove uploaded compony logo on error
    }
    return response.somethingErrorMsgResponse(res, 403, MESSAGE.EMAIL_ALREADY_REGISTERED);
  }
  // new password
  const temp_password = _value(10);
  const password = bcrypt.hashSync(temp_password, 16);
  const secquenceCount = await User.find({}).count() + 1;
  
  const user = new User({
    first_name,
    last_name,
    email:email.toLowerCase(),
    mobile_no,
    city,
    state,
    country,
    zip_code,
    company_name,
    status,
    address1,
    address2,
    password,
    role: RPC_MANAGER,
    user_id: RPC_MANAGER + '_' + secquenceCount,
    company_logo: req.body.company_logo
  });
  try {
    let saveUser = await user.save();
    // let subject = "Your Password ";
    // let text = `hi ${email} <br> and your password is ${temp_password}`;
    // mail.sendMail(email, subject, text);
    const emailData = {
      msg: {
        to: email,
        from: 'Winston@roofingprojects.com', // Use the email address or domain you verified above
        template_id: EMAIL_TEMPLATES.USER_ONBOARDING_EMAIL_TEMPLATE_ID,
        dynamic_template_data:{
          "first_name": first_name,
          "last_name": last_name,
          "password": temp_password
        }
      }
    };
    mail.sendMailWithTemplate(emailData);
    // Save user module Permission to DB
    await db.insertItem(UserPermission, {
      user_id: saveUser._id,
      module: req.body.module
    })
    return response.successResponse(res, 200, saveUser, MESSAGE.RPC_ADD_SUCCESS);

  } catch (e) {
    if (req?.files?.company_logo) {
      mediaUtils.unlink(req.body.company_logo); // remove uploaded compony logo on error
    }
    return response.somethingErrorMsgResponse(res, 500, e);
  }
};

/**
 * Updates an RPC Manager in database by id
 * @param {string} id - item id
 * @param {Object} req - request object
 */
const updateRPC = async (req, res) => {
  const { first_name, last_name, email, mobile_no, city, state, country, zip_code, company_name, status, address1, address2 } = req.body;
  try {
    // For Company Logo
    if (req?.files?.company_logo) {
      req.body.company_logo = "public/uploads/" + req.files.company_logo[0].filename;
    }
    var result = await User.findByIdAndUpdate(
      (_id = req.body._id),
      {
        first_name,
        last_name,
        email,
        mobile_no,
        city,
        state,
        country,
        zip_code,
        company_name,
        status,
        address1,
        address2,
        company_logo: req.body.company_logo
      }
      , { new: true }
    );


    var userPermissionsData = await UserPermission.findOneAndUpdate({
      user_id: req.body._id,
    }, {
      module: req.body.module
    });

    if (result) {
      result.permissions = userPermissionsData ;
      return response.successResponse(res, 200, result, MESSAGE.RPC_UPDATE_SUCCESS);
    } else {
      if (req?.files?.company_logo) {
        mediaUtils.unlink(req.body.company_logo); // remove uploaded compony logo on error
      }
      return response.badRequest(res, 400, MESSAGE.RPC_NOT_FOUND);
    }
  } catch (e) {
    if (req?.files?.company_logo) {
      mediaUtils.unlink(req.body.company_logo); // remove uploaded compony logo on error
    }
    return response.somethingErrorMsgResponse(res, 500, e);
  }
};

/**
* toggle change for RPC status from database by id
* @param {string} id - user id
*/
const updateRPCstatus = async (req, res) => {
  try {
    var result = await User.findByIdAndUpdate({ _id: req.body.id }, { $set: { status: JSON.parse(req.body.status) } }, { new: true });
    if (result) {
      return response.successResponse(res, 200, result, MESSAGE.ROOF_UPDATE_SUCCESS);
    } else {
      return response.badRequest(res, 400, MESSAGE.ROOF_UPDATE_FAILD);
    }
  } catch (e) {
    return response.somethingErrorMsgResponse(res, 500, e);
  }
};

/**
 * Delete one RPC from database by id
 * @param {string} id - user id
 */
const deleteRPC = async (req, res) => {
  try {
    let id = req.query.id;
    User.findByIdAndUpdate(
      id,
      { $set: { deleted: true } },
      { new: true },

      function (err, result) {
        if (err) {
          return response.somethingErrorMsgResponse(res, 500, MESSAGE.SOMETHING_WENT_WRONG);
        } else {
          return response.successResponse(res, 200, {}, MESSAGE.RPC_DELETE_SUCCESS);
        }
      }
    );
  } catch (e) {
    return response.somethingErrorMsgResponse(e, 500, MESSAGE.SOMETHING_WENT_WRONG);
  }
};

const searchRPCManager = async (req, res) => {
  if (req.query.value) {
    query = [
      { first_name: { $regex: req.query.value, $options: "i" } },
      { last_name: { $regex: req.query.value, $options: "i" } },
      { email: { $regex: req.query.value, $options: "i" } },
    ];
  }
  const user = await User.find(
    { $or: query, role: { $eq: "RPC_MANAGER" } },
    { password: 0 }
  )
    .limit(10)
    .skip(0);
  return res.status(200).send({
    data: user.length > 0 ? user : MESSAGE.SEARCH_USER_NOT_FOUND,
  });
};


/**
 * Get one RPC from database by id
 * @param {string} id - user id
 */
const viewRPC = async (req, res) => {
  const USER_ID = req.query.id;
  try {
    const user = await User.findOne({ _id: USER_ID }).exec();
    const usermodules = await UserPermission.findOne({ user_id: USER_ID }).exec();
    return response.successResponse(res, 200, { user, usermodules });
  } catch (e) {
    return response.somethingErrorMsgResponse(res, 500, e);
  }
};

module.exports = {
  addRPC,
  getAll_RPC,
  updateRPC,
  deleteRPC,
  searchRPCManager,
  updateRPCstatus,
  viewRPC,
};
