const response = require("../../constants/response");
const MESSAGE = require("../../constants/message");
const { ProjectCategory } = require("../../models");
const db = require("../../services/db");
const slugify = require("slugify");
const { getLoggedInUser } = require("../../utils/jwtToken");

// addNewCategory
const addCategory = async (req, res) => {
    console.log("Category Body data", req.body)
    // Validate request
    if (!(req.body.name)) {
        return response.badRequest(res, 400, MESSAGE.MANDATORY_CAT_INPUT_REQUIRED);
    }
    const authHeader = req.headers.authorization;
    let token = authHeader.split("Bearer ")[1];
    const loggedInuser = await getLoggedInUser(token);

    if (loggedInuser && loggedInuser.payload && loggedInuser.payload._id) {
        // check if category already exist
        // Validate if category exist in our database
        const oldCat = await ProjectCategory.findOne({ 'name': req.body.name ,'parent': null });

        if (oldCat) {
            return response.badRequest(res, 409, MESSAGE.CAT_ALREADY_EXIST);
        }
        // Create a category
        const catData = new ProjectCategory({
            name: req.body.name,
            created_by: loggedInuser.payload._id,
            slug: slugify((req.body.name).toLowerCase())
        });

        // Save Category in the database
        catData.save()
            .then(data => {
                return response.successResponse(res, 200, data, MESSAGE.CATEGORY_ADD_SUCCESS);
            }).catch(err => {
                return response.badRequest(res, 500, err);
            });
    }else {
        // Throw error
        return response.badRequest(res, 400, "Unauthorized Access!");
    }
};

// addSubCategory
const addSubCategory = async (req, res) => {
    // Validate request
    if (!(req.body.name && req.body.category_id)) {
        return response.badRequest(res, 400, MESSAGE.MANDATORY_SUBCAT_INPUT_REQUIRED);
    }
    const authHeader = req.headers.authorization;
    let token = authHeader.split("Bearer ")[1];
    const loggedInuser = await getLoggedInUser(token);

    if (loggedInuser && loggedInuser.payload && loggedInuser.payload._id) {
        // check if category exist or not
        const parentCategoryDetails = await ProjectCategory.findOne({ '_id': req.body.category_id ,'parent': null });
        if (parentCategoryDetails == null) {
            return response.badRequest(res, 404, MESSAGE.CAT_NOT_EXIST);
        }
        // Check if Sub category exist or not
        const oldSubCat = await ProjectCategory.findOne({ 'name': req.body.name,'parent': req.body.category_id });
        if (oldSubCat) {
            return response.badRequest(res, 409, MESSAGE.SUBCAT_ALREADY_EXIST);
        }
        // Create a sub category
        const catData = new ProjectCategory({
            name: req.body.name,
            created_by: loggedInuser.payload._id,
            parent: req.body.category_id,
            slug: slugify((req.body.name).toLowerCase())
        });

        // Save Category in the database
        catData.save()
            .then(data => {
                return response.successResponse(res, 200, data, MESSAGE.SUBCATEGORY_ADD_SUCCESS);
            }).catch(err => {
                return response.badRequest(res, 500, err);
            });
    } else {
        // Throw error
        return response.badRequest(res, 400, "Unauthorized Access!");
    }

};

// Update a Category OR sub category by unique ID
const update = async (req, res) => {
    // Validate request
    if (!(req.body.name)) {
       return response.badRequest(res, 400, req.body.category_id != null ? MESSAGE.MANDATORY_SUBCAT_INPUT_REQUIRED : MESSAGE.MANDATORY_CAT_INPUT_REQUIRED);
    }
    // Validate unique ID in request
    if (!(req.body._id)) {
        return response.badRequest(res, 400, MESSAGE.REF_ID_REQUIRED);
     }
    const authHeader = req.headers.authorization;
    let token = authHeader.split("Bearer ")[1];
    const loggedInuser = await getLoggedInUser(token);

    // Find record and update it with the request body
    ProjectCategory.findOneAndUpdate({ _id: req.body._id, created_by: loggedInuser.payload._id }, {
        name: req.body.name,
        slug: slugify((req.body.name).toLowerCase())
    }, { new: true })
        .then(data => {
            if (!data) {
                return response.badRequest(res, 404, MESSAGE.CAT_NOT_EXIST);
            }
            return response.successResponse(res, 200, data, data.parent != null ? MESSAGE.SUBCATEGORY_UPDATE_SUCCESS : MESSAGE.CATEGORY_UPDATE_SUCCESS);
        }).catch(err => {
            if (err.kind === 'ObjectId') {
                return response.badRequest(res, 404, MESSAGE.CAT_NOT_EXIST);
            }
            console.log(err)
            return response.badRequest(res, 500, err);
        });
};

// Get all DB categories
const getAllCategories = async (req, res) => {
    // try {
    //     const categoryData = await ProjectCategory.find({ 'deleted': false, parent: { $eq: null } }).sort([["updatedAt",-1]]);
    //     return response.successResponse(res, 200, categoryData, MESSAGE.CAT_SUBCAT_LIST_DATA_SUCCESS);
    // } catch (error) {
    //     console.log(error);
    //     return response.somethingErrorMsgResponse(res, 500, error);
    // }

    try {
        let query = await db.checkQueryString(req.query)
        query.parent = { $eq: null };
        console.log("query", JSON.stringify(query))
        return response.successResponse(res, 200, await db.getItems(req, ProjectCategory, query), MESSAGE.CAT_SUBCAT_LIST_DATA_SUCCESS);
      } catch (error) {
        return response.somethingErrorMsgResponse(res, 500, error);
      }
};

// Get all DB subcategories
const getAllSubCategories = async (req, res) => {
    // try {
    //     const categoryData = await ProjectCategory.find({ 'deleted': false, parent: { $ne: null } }).sort([["updatedAt",-1]]);
    //     return response.successResponse(res, 200, categoryData, MESSAGE.CAT_SUBCAT_LIST_DATA_SUCCESS);
    // } catch (error) {
    //     console.log(error);
    //     return response.somethingErrorMsgResponse(res, 500, error);
    // }

    try {
        let query = await db.checkQueryString(req.query)
        query.parent = { $ne: null };
        console.log("query", JSON.stringify(query))
        return response.successResponse(res, 200, await db.getItems(req, ProjectCategory, query), MESSAGE.CAT_SUBCAT_LIST_DATA_SUCCESS);
      } catch (error) {
        return response.somethingErrorMsgResponse(res, 500, error);
      }
};

// Get sub categories by Cat id
const getSubCategoriesByCatId = async (req, res) => {
    // try {
    //     const categoryData = await ProjectCategory.find({ 'deleted': false, 'parent': req.params.cat_id }).sort([["updatedAt",-1]]);
    //     return response.successResponse(res, 200, categoryData, MESSAGE.CAT_SUBCAT_LIST_DATA_SUCCESS);
    // } catch (error) {
    //     console.log(error);
    //     return response.somethingErrorMsgResponse(res, 500, error);
    // }

    try {
        let query = await db.checkQueryString(req.query)
        query.parent = req.params.cat_id;
        console.log("query", JSON.stringify(query))
        return response.successResponse(res, 200, await db.getItems(req, ProjectCategory, query), MESSAGE.CAT_SUBCAT_LIST_DATA_SUCCESS);
      } catch (error) {
        return response.somethingErrorMsgResponse(res, 500, error);
      }
};

// Delete a category & associated sub categories with the specified id in the request
const deleteCategory = async (req, res) => {
      // Delete category and its subcategories
      if (req.params && req.params.id) {
        ProjectCategory.delete({_id:req.params.id})
            .then(responseData => {
                if (!responseData) {
                    return response.badRequest(res, 404, MESSAGE.CAT_NOT_EXIST);
                }
                // delete all associated sub categories
                ProjectCategory.delete({ parent: req.params.id }, function (err) {
                    if (err) {
                        return response.badRequest(res, 500, err);
                    }
                    return response.successResponse(res, 200, MESSAGE.CATEGORY_DELETE_SUCCESS)
                });
            }).catch(err => {
                if (err.kind === 'ObjectId' || err.name === 'NotFound') {
                    return response.badRequest(res, 404, MESSAGE.CAT_NOT_EXIST);
                }
                return response.badRequest(res, 500, err);
            });
    }else{
        return response.badRequest(res, 500, MESSAGE.SOMETHING_WENT_WRONG);
    }
};

// Delete a sub category with the specified id in the request
const deleteSubCategory = async (req, res) => {
    // Delete subcategory 
    if (req.params && req.params.id) {
      ProjectCategory.delete({_id:req.params.id})
          .then(responseData => {
              if (!responseData) {
                  return response.badRequest(res, 404, MESSAGE.CAT_NOT_EXIST);
              }
              return response.successResponse(res, 200, MESSAGE.SUBCATEGORY_DELETE_SUCCESS);
          }).catch(err => {
              if (err.kind === 'ObjectId' || err.name === 'NotFound') {
                  return response.badRequest(res, 404, MESSAGE.CAT_NOT_EXIST);
              }
              return response.badRequest(res, 500, err);
          });
  }else{
      return response.badRequest(res, 500, MESSAGE.SOMETHING_WENT_WRONG);
  }
};

module.exports = {
    addCategory,
    addSubCategory,
    update,
    getAllCategories,
    getAllSubCategories,
    getSubCategoriesByCatId,
    deleteCategory,
    deleteSubCategory
}
