const response = require("../../constants/response");
const MESSAGE = require("../../constants/message");
const EMAIL_TEMPLATES = require("../../constants/email_templates_id");
const mail = require("../../utils/notify");

const { projectBidSubmission, Project , NotificationsSettings, User} = require("../../models");
const db = require("../../services/db");
const { getLoggedInUser } = require("../../utils/jwtToken");
const pushNotification = require("../../push_notification");


// Submit Bid
const submitBid = async (req, res) => {
    // Validate request
    if (!req.body.project_id) {
        return response.errorMessageResponse(res, 500, MESSAGE.PROJECT_ID_REQUIRED);
    }
    if (!req.body.line_items) {
        return response.badRequest(res, 400, MESSAGE.BID_SPECIFICATION_ITEMS_REQUIRED);
    }
    if (req.body.line_items && req.body.line_items.length == 0) {
        return response.badRequest(res, 400, MESSAGE.EMPTY_LINEITEMS_CONTENT_ERROR);
    }
    const authHeader = req.headers.authorization;
    let token = authHeader.split("Bearer ")[1];
    const loggedInuser = await getLoggedInUser(token);
    const userId = loggedInuser.payload._id;
    var contractorDetails = await User.findOne({ _id: userId, deleted:false});

    // check if project is OPEN and accepting Bids
    var isPorjectOpen = await Project.findOne({ _id: req.body.project_id, deleted:false , project_status:'open', status:true});

    if(!isPorjectOpen){
        return response.badRequest(res, 400, "We are unable to submit your bid.Please check the selected project. Either project is closed & not active or reference provided is not correct.");
    }
    const userHasAlreadySubmitBid = await projectBidSubmission.find({ submitted_by:userId , project_id : req.body.project_id });
    // check if user already submitted the Bid on the project
    if (userHasAlreadySubmitBid.length > 0) {
        return response.errorMessageResponse(res, 500, MESSAGE.MULTIPLE_BID_SUBMIT_NOT_ALLOWED)
      }
    // Create a bid form template
    const bidFormData = new projectBidSubmission({
        project_id: req.body.project_id,
        submitted_by: userId,
        selected_manufacturer: req.body.selected_manufacturer,
        total_bid_amount: req.body.total_bid_amount,
        total_bid_amount_in_words: req.body.total_bid_amount_in_words,
        line_items: req.body.line_items ? req.body.line_items : [],
        expected_start_date: req.body.expected_start_date,
        days_to_complete: req.body.days_to_complete
    });

    // Save record in the database
    bidFormData.save()
        .then(async data => {
            const projectData = await Project.findOne({ _id: req.body.project_id });
            console.log("projectData", projectData)
            projectData.bids_submission.push(data._id);
            projectData.participating_bidders.push(userId);
            let results = await db.updateItem(req.body.project_id, Project, projectData);
            const rpcManagers = await User.find({ role: 'RPC_MANAGER', deleted:false})
            .select("-__v -createdAt -updatedAt -password -otp -mobile_no -phone_no -profile_pic -city -state -country -zip_code");
            var RPCManagersEmails = await rpcManagers.map(function (details) {
                    return details['email'];
                });

            const emailData = {
                msg: {
                  to: RPCManagersEmails, // Notify to all rpc managers that bid submitted to a project
                  from: 'Winston@roofingprojects.com', // Use the email address or domain you verified above
                  template_id: EMAIL_TEMPLATES.BID_SUBMISSION_TEMPLATE_ID,
                  dynamic_template_data:{
                    "contractor_first_name":contractorDetails.first_name,
                    "contractor_last_name":contractorDetails.last_name,
                    "project_name":projectData ? projectData.project_title : 'Project Details Not Available',
                    "location": projectData.address +" ," + projectData.city +", "+ projectData.state +", "+ projectData.country +" "+ projectData.zip_code,
                    "size": projectData.square_footage,
                    "total_bid_amount": req.body.total_bid_amount,
                    "total_bid_amount_in_words": req.body.total_bid_amount_in_words,
                    "expected_start_date": req.body.expected_start_date,
                    "days_to_complete": req.body.days_to_complete
                    }
                }
              };
              const contractorEmailData = {
                msg: {
                  to: contractorDetails.email, // Notify to all contractor for submissio success
                  from: 'Winston@roofingprojects.com', // Use the email address or domain you verified above
                  template_id: EMAIL_TEMPLATES.BID_SUBMISSION_SUCCESS_TEMPLATE_ID,
                  dynamic_template_data:{
                    "first_name":contractorDetails.first_name,
                    "last_name":contractorDetails.last_name,
                    "project_name":projectData ? projectData.project_title : 'Project Details Not Available',
                    "location": projectData.address +" ," + projectData.city +", "+ projectData.state +", "+ projectData.country +" "+ projectData.zip_code,
                    "size": projectData.square_footage,
                    "total_bid_amount": req.body.total_bid_amount,
                    "total_bid_amount_in_words": req.body.total_bid_amount_in_words,
                    "expected_start_date": req.body.expected_start_date,
                    "days_to_complete": req.body.days_to_complete
                    }
                }
              };
              mail.sendMailWithTemplate(emailData); // to all rpc managers
              mail.sendMailWithTemplate(contractorEmailData); // to contractor

            return response.successResponse(res, 200, data, MESSAGE.BID_SUBMIT_SUCCESS);
        }).catch(err => {
            console.log(err)
            return response.errorMessageResponse(res, 500, err);
        });
};

// Get All my Bids or filter by project ID
const getMyBidDetails = async (req, res) => {
    try {
        const authHeader = req.headers.authorization;
        let token = authHeader.split("Bearer ")[1];
        const loggedInuser = await getLoggedInUser(token);
        const userId = loggedInuser.payload._id;
        const options = [
            {  
                path: 'submitted_by',
                select: ['first_name', 'last_name', 'email', 'city', 'state', 'zip_code', 'street_address', 'mobile_no','company_name', 'profile_pic']
            },
            {
                path: 'selected_manufacturer',
                select: ['first_name', 'last_name', 'email', 'city', 'state', 'zip_code', 'street_address', 'mobile_no','company_name', 'profile_pic']
            }
        ];
        const myBidDetails = await db.getItemByParams(projectBidSubmission, { submitted_by: userId, project_id: req.params.project_id }, options);
        return response.successResponse(res, 200, myBidDetails, MESSAGE.MY_BID_DETAILS_SUCCESS);
    } catch (e) {
        return response.somethingErrorMsgResponse(res, 500, e);
    }
};

// Get All Bid submissions on a Project
const getAllBids = async (req, res) => {
    try {
        const bidSubmissions = await projectBidSubmission.find({ project_id: req.params.project_id })
        .populate({
            path: 'submitted_by',
            select: ['first_name', 'last_name', 'email', 'city', 'state', 'zip_code', 'street_address', 'mobile_no','company_name', 'profile_pic']
        })
        .populate({
            path: 'selected_manufacturer',
            select: ['first_name', 'last_name', 'email', 'city', 'state', 'zip_code', 'street_address', 'mobile_no','company_name', 'profile_pic']
        })
        .populate({
            path: 'project_id',
            select: ['project_title']
        })
        .sort([["updatedAt",-1]]);
        return response.successResponse(res, 200, bidSubmissions, MESSAGE.MY_BID_SUBMISSION_LIST);
    } catch (e) {
        return response.somethingErrorMsgResponse(res, 500, e);
    }
};

// Update Project Bid form Template
const updateBid = async (req, res) => {
    // Validate request
    if (!req.body.project_id) {
        return response.errorMessageResponse(res, 400, MESSAGE.PROJECT_ID_REQUIRED);
    }
    if (!req.body.line_items) {
        return response.badRequest(res, 400, MESSAGE.BID_SPECIFICATION_ITEMS_REQUIRED);
    }
    if (req.body.line_items && req.body.line_items.length == 0) {
        return response.badRequest(res, 400, MESSAGE.EMPTY_LINEITEMS_CONTENT_ERROR);
    }
    const authHeader = req.headers.authorization;
    let token = authHeader.split("Bearer ")[1];
    const loggedInuser = await getLoggedInUser(token);
    const userId = loggedInuser.payload._id;

    const payload = {
        project_id: req.body.project_id,
        selected_manufacturer: req.body.selected_manufacturer,
        submitted_by: userId,
        total_bid_amount: req.body.total_bid_amount,
        total_bid_amount_in_words: req.body.total_bid_amount_in_words,
        line_items: req.body.line_items ? req.body.line_items : [],
        expected_start_date: req.body.expected_start_date,
        days_to_complete: req.body.days_to_complete
    }
    let results = await db.updateItem(req.body._id, projectBidSubmission, payload)

    if (results) {
        return response.successResponse(res, 200, results, MESSAGE.BID_UPDATE_SUCCESS);
    } else {
        return response.badRequest(res, 400, MESSAGE.BID_UPDATE_FAILED);
    }
};

const awardBid = async (req, res) => {
    const { bid_id } = req.body;
    if (!req.body.bid_id) {
        return response.badRequest(res, 400, MESSAGE.BID_REF_ID_REQUIRED);
    }
    try {
      const bidDetails = await projectBidSubmission.find({_id: bid_id,}).exec();
      if (Array.isArray(bidDetails) && bidDetails.length == 0) {
        return response.conflictErrorMsgResponse(res, 409, MESSAGE.BID_REF_ID_NOT_FOUND);
      }
      const bids = await projectBidSubmission.find({
        project_id: bidDetails[0].project_id,
        is_bid_winner: true
      }).exec()
      if (Array.isArray(bids) && bids.length > 0) {
        return response.conflictErrorMsgResponse(res, 409, MESSAGE.BID_ALREADY_AWARDED);
      }
      const result = await projectBidSubmission.findByIdAndUpdate({_id:bid_id}, { is_bid_winner: true, bid_status:'CLOSED' }, { new: true });
      if (result) {
        const updateAllBidStatus = await projectBidSubmission.updateMany(
            { project_id: bidDetails[0].project_id },
            { $set: { bid_status : 'CLOSED' } },
            {multi: true}
        );
        const projectDataUpdate = await Project.findByIdAndUpdate({_id:bidDetails[0].project_id}, { statue: false, project_status:'CLOSE' }, { new: true });
       
        const NotificationEnabled = await NotificationsSettings.findOne({ type: 'PROJECT_ACTIVATION', enabled: true }).exec();
        if(NotificationEnabled){
            // send the push notifications to all bidders
            const toUsersId = [...projectDataUpdate.participating_bidders]
            const NotificationData = {
                type: "BID_AWARDED",
                title: "BID is awarded for a project - " + projectDataUpdate.project_title +" and Project is closed.",
                notification_content: "Dear Bidder , We would like to update you that a Bid is awarded for the project "+ projectDataUpdate.project_title +" . and the project is marked as closed. Thanks, Roofing Team ."
            }
            await pushNotification.sendNotification(req, res, toUsersId, NotificationData, req.body);
        }
        // send Email Notifications to bidders
        const projectDetails = await Project.findOne({ _id: bidDetails[0].project_id })
        .populate([{path:'participating_bidders',select:['first_name', 'last_name', 'email']}]);
        var participatingBiddersEmails = await (projectDetails.participating_bidders).map(function (details) {
            return details['email'];
        });
        const winnerDetails = await User.findOne({ _id: bidDetails[0].submitted_by });

        var filteredBiddersEmails = participatingBiddersEmails.filter(function(value, index, arr){ 
            return (value != winnerDetails.email);
        });
        const emailData = {
            msg: {
              to: filteredBiddersEmails, // Notify to all bidders
              from: 'Winston@roofingprojects.com', // Use the email address or domain you verified above
              template_id: EMAIL_TEMPLATES.PROJECT_AWARDED_NOTIFY_BIDDERS_TEMPLATE_ID,
              dynamic_template_data:{
                "project_name":projectDetails ? projectDetails.project_title : 'Project Details Not Available',
                "location": projectDetails.address +" ," + projectDetails.city +", "+ projectDetails.state +", "+ projectDetails.country +" "+ projectDetails.zip_code,
                "size": projectDetails.square_footage
                }
            }
          };
          const winnerEmailData = {
            msg: {
              to: winnerDetails.email, // Notify to all bidders
              from: 'Winston@roofingprojects.com', // Use the email address or domain you verified above
              template_id: EMAIL_TEMPLATES.PROJECT_AWARDED_NOTIFY_WINNER_TEMPLATE_ID,
              dynamic_template_data:{
                "first_name": winnerDetails.first_name,
                "last_name":winnerDetails.last_name,
                "project_name":projectDetails ? projectDetails.project_title : 'Project Details Not Available',
                "location": projectDetails.address +" ," + projectDetails.city +", "+ projectDetails.state +", "+ projectDetails.country +" "+ projectDetails.zip_code,
                "size": projectDetails.square_footage
                }
            }
          };
        mail.sendMailWithTemplate(emailData); // to all bidders except winner regarding bid awarded
        mail.sendMailWithTemplate(winnerEmailData); // to all bidders except winner regarding bid awarded
        
        return response.successResponse(res, 200, result, MESSAGE.BID_AWARDED_SUCCESS);
      } else {
        return response.badRequest(res, 400, MESSAGE.BID_AWARD_FAILED);
      }
    } catch (e) {
      return response.somethingErrorMsgResponse(res, 500, e);
    }
  }

module.exports = {
    submitBid,
    getMyBidDetails,
    updateBid,
    getAllBids,
    awardBid
}
