const response = require("../../constants/response");
const MESSAGE = require("../../constants/message");
const EMAIL_TEMPLATES = require("../../constants/email_templates_id");
const mail = require("../../utils/notify");

const { Project, projectBidInvitation, User } = require("../../models");
const db = require("../../services/db");
const { getLoggedInUser } = require("../../utils/jwtToken");
const pushNotification = require("../../push_notification");


// Send Bid Inviation to contractor(s)
const sendBidInvitation = async (req, res) => {
    // Validate request
    if (!req.body.project_id) {
        return response.errorMessageResponse(res, 500, MESSAGE.PROJECT_ID_REQUIRED);
    }
    if (!(req.body.not_registered_contractors && req.body.contractor_ids)) {
        return response.badRequest(res, 400, "Please contractor email details to send the bid invitation.");
    }
    
    const authHeader = req.headers.authorization;
    let token = authHeader.split("Bearer ")[1];
    const loggedInuser = await getLoggedInUser(token);
    const userId = loggedInuser.payload._id;
    
    // Get project details
    const projectDetails = await Project.findOne({ _id: req.body.project_id})
    .populate([{path:'buildingId'}]);
    // set push notification data
    const projectName = projectDetails.project_title ;
    const square_footage = "approximately  "+ projectDetails.square_footage +" square feet";
    const bid_period = "The bid period is "+ projectDetails.bid_start_date  +" to "+ projectDetails.bid_end_date ;
    const htmlTextMsg ='<p> Dear,  </p> <p> <strong> <span style="color:green"> Bid Invitation </span> </strong> You are invited to bid on the project with below details</p><p> <h5>'+ projectName +'</h5> <p>'+ square_footage +'</p></p> <p>'+ bid_period +'</p> <br> Thanks.'
    const subj ='Project Bid Invitation- RoofingProjects.com';

    // Create save Payload
    const payload = new projectBidInvitation({
        project_id: req.body.project_id,
        invitation_title: subj,
        invitation_content: htmlTextMsg,
        contractor_ids: req.body.contractor_ids,
        not_registered_contractors: req.body.not_registered_contractors,
        createdBy: userId,
    });

    // Save record in the database
    payload.save()
        .then(async data => {

            const BidInvitationdetails = await projectBidInvitation.findOne({ _id: data._id })
            .populate([{path:'contractor_ids',select:['first_name', 'last_name', 'email']}]);

            const NotificationData = {
                type: "BID_INVITATION",
                title: "BID INIVITATION- "+ subj,
                notification_content: htmlTextMsg
            }
            //send Push Notification to  Registered Contractor
            if(req.body.contractor_ids && req.body.contractor_ids.length > 0){
                // Send push notification to each REGISTERED contractors
                BidInvitationdetails.contractor_ids.forEach(async element => {
                    await pushNotification.sendNotification(req, res, element._id, NotificationData, req.body);
                    // Now send emails to all contractors one by one
                    const emailData = {
                        msg: {
                          to: element.email,
                          from: 'Winston@roofingprojects.com', // Use the email address or domain you verified above
                          template_id: EMAIL_TEMPLATES.PROJECT_BID_INVITATION,
                          dynamic_template_data:{
                            "first_name":element.first_name ? element.first_name : 'Guest',
                            "last_name":element.last_name ? element.last_name : 'User',
                            "project_name": projectDetails.project_title,
                            "location": projectDetails.address +" ," + projectDetails.city +", "+ projectDetails.state +", "+ projectDetails.country +" "+ projectDetails.zip_code,
                            "size": projectDetails.square_footage,
                            "manufacturers": projectDetails.manufacturers,
                            "bid_start":projectDetails.bid_start_date,
                            "bid_end":projectDetails.bid_end_date,
                            "building_name": projectDetails.buildingId.name
                        }
                        }
                      };
                    mail.sendMailWithTemplate(emailData);
                
                });
            }

            // Send email to Not Registered users
            if(req.body.not_registered_contractors && req.body.not_registered_contractors.length > 0){
                // Now send email to all Not Registered user
                const emailData = {
                    msg: {
                      to: req.body.not_registered_contractors,
                      from: 'Winston@roofingprojects.com', // Use the email address or domain you verified above
                      template_id: EMAIL_TEMPLATES.PROJECT_BID_INVITATION,
                      dynamic_template_data:{
                        "first_name": 'Guest',
                        "last_name": 'User',
                        "project_name": projectDetails.project_title,
                        "location": projectDetails.address +" ," + projectDetails.city +", "+ projectDetails.state +", "+ projectDetails.country +" "+ projectDetails.zip_code,
                        "size": projectDetails.square_footage,
                        "manufacturers": projectDetails.manufacturers,
                        "bid_start":projectDetails.bid_start_date,
                        "bid_end":projectDetails.bid_end_date,
                        "building_name": projectDetails.buildingId.name
                    }
                    }
                  };
                mail.sendMailWithTemplate(emailData);
            }

            return response.successResponse(res, 200, data, "Bid Invitation has been sent successfully.");
        }).catch(err => {
            return response.errorMessageResponse(res, 500, err);
        });

};

const getMyInvitationRequests = async (req,res) => {
    try {
        const authHeader = req.headers.authorization;
        let token = authHeader.split("Bearer ")[1];
        const loggedInuser = await getLoggedInUser(token);
        const userId = loggedInuser.payload._id;
        const result = await projectBidInvitation.find({contractor_ids : userId})
        .populate([{path:'project_id'}]);
        return response.successResponse(res, 200, result, "Bid Invitation List reterived successfully.");
    } catch(error) {
        return response.somethingErrorMsgResponse(res, 500, error);
    }
}

const getAllInvitationByProjectId = async (req,res) => {
    try {
        if (!req.body.project_id) {
            return response.errorMessageResponse(res, 500, MESSAGE.PROJECT_ID_REQUIRED);
        }
        const result = await projectBidInvitation.find({project_id : req.body.project_id})
        .populate([{path:'contractor_ids',select:['first_name', 'last_name', 'email']}])
        .populate([{path:'project_id'},{path:'contractor_id'}]);
        return response.successResponse(res, 200, result, "Project Bid Invitation requests reterived successfully.");
    } catch(error) {
        return response.somethingErrorMsgResponse(res, 500, error);
    }
}

const acceptInvitation = async (req,res) => {
    try {
        // Validate request
        if (!req.body.id) {
            return response.errorMessageResponse(res, 500, "Please provide the correct invitation request id.");
        }
        let filter = { _id: req.query.id };
        let update = {"$set":{ user_response:'ACCEPTED'}};
        let isAlreadyAccepted = await Invitation.findOne({ _id: req.query.id, user_response : 'ACCEPTED' });
        if(isAlreadyAccepted){
            return response.successResponse(res, 200, {}, "You have already accepted this Invite.");
        }
        const result = await Invitation.findOneAndUpdate(filter,update,{new:true});
        if(result)
            return response.successResponse(res, 200, result, MESSAGE.BID_INVITATION_ACCEPTED);
        else
            return response.successResponse(res, 200,{}, "Error while saving your response. Please try again." );
    } catch(error) {
        return response.somethingErrorMsgResponse(res, 500, error);
    }
}

const rejectInvitation = async (req,res) => {
    try {
        // Validate request
        if (!req.body.id) {
            return response.errorMessageResponse(res, 500, "Please provide the correct invitation request id.");
        }
        let filter = { _id: req.query.id };
        let update = {"$set":{ user_response:'REJECTED'}};
        let isAlreadyAccepted = await Invitation.findOne({ _id: req.query.id, user_response : 'ACCEPTED' });
        if(isAlreadyAccepted){
            return response.successResponse(res, 200, {}, "You have already rejected this Invite.");
        }
        const result = await Invitation.findOneAndUpdate(filter,update,{new:true});
        if(result)
            return response.successResponse(res, 200, result, "Invitation Rejection submitted.");
        else
            return response.successResponse(res, 200,{}, "Error while saving your response. Please try again." );
    } catch(error) {
        return response.somethingErrorMsgResponse(res, 500, error);
    }
}

// API for sending Email to single or multiple users
// pass array like---> "email":["testmail@gmail.com","tastemail1@gmail.com"]
const sendEmailtoBidders = async (req, res) => {
    try {
        if (!req.body.email) {
            return response.errorMessageResponse(res, 500, "Please select the bidders.");
        }
        if (!req.body.subject) {
            return response.errorMessageResponse(res, 500, "Please provide the subject of email.");
        }
        if (!req.body.message) {
            return response.badRequest(res, 400, "Please provide the email content.");
        }
        if (req.body.message && req.body.subject) {
            if((req.body.subject).trim() ==''  || (req.body.message).trim() == ''){
                return response.badRequest(res, 400, "Email subject and content cannot be empty.");
            }
        }
        if (req.body.email && (req.body.email).length == 0) {
            return response.errorMessageResponse(res, 500, "Email Receipent cant be empty.");
        }
      let email = req.body.email;
      let subject = req.body.subject;
      let text = req.body.message;
      console.log(req.body)
      mail.sendMail(email, subject, text);
    //   const emailData = {
    //     msg: {
    //       to: 'shivam@yopmail.com',
    //       from: 'Winston@roofingprojects.com', // Use the email address or domain you verified above
    //       template_id: 'd-98cc1aa8ac264acb8bb3a9a291e1607d',
    //       dynamic_template_data:{
    //         "first_name":"User",
    //         "last_name":"",
    //         "project_name": "ACME roof replacement",
    //         "location": "Acme Market - 4400 South Broad St, Yardville, NJ 08620",
    //         "size": "56,000",
    //         "manufacturers": "Sarnafil, Duro-Last, Carlisle, Versico, Johns Manville, and GAF",
    //         "bid_start":"03/16/2022",
    //         "bid_end":"03/28/2022 5:00 PM EST"
    //     }
    //     }
    //   };
    //   mail.sendMailWithTemplate(emailData);

      return response.successResponse(res, 200,email,MESSAGE.BID_EMAIL_SENT_TO_BIDDER);
    } catch (e) {
        console.log("error", e)
      return response.somethingErrorMsgResponse(res, 500, e);
    }
  }

module.exports = {
    sendBidInvitation,
    getMyInvitationRequests,
    getAllInvitationByProjectId,
    acceptInvitation,
    rejectInvitation,
    sendEmailtoBidders
}
