const response = require("../../constants/response");
const MESSAGE = require("../../constants/message");
const EMAIL_TEMPLATES = require("../../constants/email_templates_id");

const { User } = require("../../models");
const { emailStatus } = require("../../utils/getUser");
const { OWNER } = require("../../constants/roles");
const bcrypt = require("bcryptjs");
const { _value } = require("../../utils/getRandom");
const db = require("../../services/db");
const mail = require("../../../MEAN-Roofing/utils/notify");
const mediaUtils = require("../../utils/unlinkMedia");

/**
 * Creates a new Owner in database
 * @param {Object} req - request object
 */
const add = async (req, res) => {
  const {
    first_name,
    last_name,
    email,
    mobile_no,
    city,
    state,
    country,
    zip_code,
    company_name,
    status,
    address1,
    address2,
    owner_type
  } = req.body;

  // For Company Logo
  if (req?.files?.company_logo) {
    req.body.company_logo = "public/uploads/" + req.files.company_logo[0].filename;
  }
  // if mail exist then return message
  if (await emailStatus(email)) {
    if (req?.files?.company_logo) {
      mediaUtils.unlink(req.body.company_logo); // remove uploaded compony logo on error
    }
    return response.somethingErrorMsgResponse(res, 403, MESSAGE.EMAIL_ALREADY_REGISTERED);
  }
  // new password
  const temp_password = _value(10);
  // Random Generate Password for User that is get by on user's email......
  const password = bcrypt.hashSync(temp_password, 16);
  console.log(temp_password, "temp_password"); //@GUNJAN/@PRASHANT/@SHANTANU remove after email integration
  // This is the only permissions for user
  const secquenceCount = await User.find({}).count() + 1;

  // End User Permissions
  const user = new User({
    first_name,
    last_name,
    email,
    mobile_no,
    city,
    state,
    country,
    zip_code,
    company_name,
    status,
    address1,
    address2,
    password,
    role: OWNER,
    owner_type,
    createdBy: req.id,
    user_id: OWNER + '_' + secquenceCount,
    company_logo: req.body.company_logo
  });
  try {
    const saveUser = await user.save();
    // let subject = "Your Password ";
    // let text = `hi ${email} <br> and your password is ${temp_password}`;
    // mail.sendMail(email, subject, text);
    const emailData = {
      msg: {
        to: email,
        from: 'Winston@roofingprojects.com', // Use the email address or domain you verified above
        template_id: EMAIL_TEMPLATES.USER_ONBOARDING_EMAIL_TEMPLATE_ID,
        dynamic_template_data:{
          "first_name": first_name,
          "last_name": last_name,
          "password": temp_password
        }
      }
    };
    mail.sendMailWithTemplate(emailData);
    return response.successResponse(res, 200, saveUser, MESSAGE.OWNER_ADD_SUCCESS);
  } catch (e) {
    if (req?.files?.company_logo) {
      mediaUtils.unlink(req.body.company_logo); // remove uploaded compony logo on error
    }
    return response.somethingErrorMsgResponse(res, 500, e);
  }
};

/**
 * Updates an Owner in database by id
 * @param {string} id - item id
 * @param {Object} req - request object
 */
const update = async (req, res) => {
  const {
    first_name,
    last_name,
    email,
    mobile_no,
    city,
    state,
    country,
    zip_code,
    company_name,
    status,
    address1,
    address2
  } = req.body;

  try {
    // For Company Logo
    if (req?.files?.company_logo) {
      req.body.company_logo = "public/uploads/" + req.files.company_logo[0].filename;
    }
    var result = await User.findByIdAndUpdate(
      (_id = req.body._id),
      {
        first_name,
        last_name,
        email,
        mobile_no,
        city,
        state,
        country,
        zip_code,
        company_name,
        status,
        address1,
        address2,
        updatedBy: req.id,
        company_logo: req.body.company_logo
      }
    );
    if (result) {
      return response.successResponse(res, 200, result, MESSAGE.OWNER_UPDATE_SUCCESS);
    } else {
      if (req?.files?.company_logo) {
        mediaUtils.unlink(req.body.company_logo); // remove uploaded compony logo on error
      }
      return response.badRequest(res, 400, MESSAGE.OWNER_NOT_FOUND);
    }
  } catch (e) {
    if (req?.files?.company_logo) {
      mediaUtils.unlink(req.body.company_logo); // remove uploaded compony logo on error
    }
    return response.somethingErrorMsgResponse(res, 500, e);
  }
};

/**
 * @description Function For List of all Owner with respective Buildings.
 * @param {object} req 
 * @param {object} res 
 */
const viewAll = async (req, res) => {
  try {
    let query = await db.checkQueryString(req.query)
    query.role = "OWNER" 
    req.populate = { path: 'buildings' }
    if(req?.query?.filter && !isNaN(req?.query?.filter))
    query.$or.push({mobile_no:Number(req.query.filter)})
    return response.successResponse(res, 200, await db.getItems(req, User, query), MESSAGE.OWNER_VIEW_DATA);
  } catch (error) {
    return response.somethingErrorMsgResponse(res, 500, error);
  }
}

/**
* Soft Delete one owner from database by id
* @param {string} id - owner id
*/
const deleteOwner = async (req, res) => {
  try {
    const result = await db.softDeleteItems(req.params.id, User)
    if (result) {
      return response.successResponse(res, 200, {}, MESSAGE.OWNER_DELETE_SUCCESS);
    } else {
      return response.somethingErrorMsgResponse(res, 500, MESSAGE.SOMETHING_WENT_WRONG);
    }
  } catch (e) {
    return response.somethingErrorMsgResponse(res, 500, e);
  }
}

/**
 * Get one OWNER from database by id
 * @param {string} id - user id
 */
const viewOne = async (req, res) => {
  const USER_ID = req.query.id;
  try {
    const user = await User.findOne({ _id: USER_ID }).exec();
    return response.successResponse(res, 200, user);
  } catch (e) {
    return response.somethingErrorMsgResponse(res, 500, e);
  }
};

/**
* toggle change for Owner status from database by id
* @param {string} id - owner id
*/
const updatestatus = async (req, res) => {
  const USER_ID = req.query.id;
  User.findOne({ _id: USER_ID }, (err, user) => {
    user.status = !user.status;
    user.save(function (err, updatedUser) {
      if (!err) {
        return response.successResponse(res, 200, { status: user.status }, MESSAGE.USER_STATUS_UPDATED);
      } else {
        return response.somethingErrorMsgResponse(res, 500, err);
      }
    });
  });
};

const search = async (req, res) => {
  if (req.query.value) {
    query = [
      { first_name: { $regex: req.query.value, $options: "i" } },
      { last_name: { $regex: req.query.value, $options: "i" } },
      { email: { $regex: req.query.value, $options: "i" } },
    ];
  }
  const user = await User.find(
    { $or: query, role: { $eq: "OWNER" } },
    { password: 0 }
  )
    .limit(10)
    .skip(0);
  return res.status(200).send({
    data: user.length > 0 ? user : MESSAGE.SEARCH_USER_NOT_FOUND,
  });
};

module.exports = {
  add,
  update,
  viewOne,
  deleteOwner,
  viewAll,
  updatestatus,
  search,
};
