const response = require("../../constants/response");
const MESSAGE = require("../../constants/message");
const EMAIL_TEMPLATES = require("../../constants/email_templates_id");

const { User, ContractorInfo, File, Media, Bid, Project } = require("../../models");
const { emailStatus } = require("../../utils/getUser");
const { CONTRACTOR } = require("../../constants/roles");
const bcrypt = require("bcryptjs");
const mail = require("../../utils/notify");
const db = require("../../services/db");
const { _value } = require("../../utils/getRandom");
const mediaUtils = require("../../utils/unlinkMedia");

/**
 * Creates a new Contractor in database
 * @param {Object} req - request object
 */
const addContractor = async (req, res) => {
    const {
        first_name,
        last_name,
        email,
        mobile_no,
        city,
        state,
        country,
        zip_code,
        company_name,
        status,
        address1,
        address2,
        permissions,
    } = req.body;
    // For Company Logo
    if (req?.files?.company_logo) {
        req.body.company_logo = "public/uploads/" + req.files.company_logo[0].filename;
    }
    let MailExist = await emailStatus(email);
    // if mail exist then return message
    if (MailExist) {
        if (req?.files?.company_logo) {
            mediaUtils.unlink(req.body.company_logo); // remove uploaded compony logo on error
        }
        return response.somethingErrorMsgResponse(res, 403, MESSAGE.EMAIL_ALREADY_REGISTERED);
    }
    const secquenceCount = await User.find({}).count() + 1;

    // new password
    const temp_password = _value(10);
    const password = bcrypt.hashSync(temp_password, 16);
    const user = new User({
        first_name,
        last_name,
        email,
        mobile_no,
        city,
        state,
        country,
        zip_code,
        company_name,
        status,
        address1,
        address2,
        permissions,
        password,
        role: CONTRACTOR,
        user_id: CONTRACTOR + '_' + secquenceCount,
        company_logo: req.body.company_logo
    });
    try {
        const saveContractor = await user.save();
        // let subject = "Your Password ";
        // let text = `hi ${email} <br> and your password is ${temp_password}`;
        // mail.sendMail(email, subject, text);
        const emailData = {
            msg: {
              to: email,
              from: 'Winston@roofingprojects.com', // Use the email address or domain you verified above
              template_id: EMAIL_TEMPLATES.USER_ONBOARDING_EMAIL_TEMPLATE_ID,
              dynamic_template_data:{
                "first_name": first_name,
                "last_name": last_name,
                "password": temp_password
              }
            }
          };
          mail.sendMailWithTemplate(emailData);
        return response.successResponse(res, 200, saveContractor, MESSAGE.CONTRACTOR_ADD_SUCCESS);
    } catch (e) {
        if (req?.files?.company_logo) {
            mediaUtils.unlink(req.body.company_logo); // remove uploaded compony logo on error
        }
        return response.somethingErrorMsgResponse(res, 500, e);
    }
}


/**
 * Update a Contractor in database by id
 * @param {string} id - item id
 * @param {Object} req - request object
 */
const updateContractor = async (req, res) => {
    const {
        first_name,
        last_name,
        email,
        mobile_no,
        city,
        state,
        country,
        zip_code,
        company_name,
        status,
        address1,
        address2,
        permissions,
    } = req.body;

    try {
        // For Company Logo
        if (req?.files?.company_logo) {
            req.body.company_logo = "public/uploads/" + req.files.company_logo[0].filename;
        }
        var result = await User.findByIdAndUpdate(
            (_id = req.body._id),
            {
                first_name,
                last_name,
                email,
                mobile_no,
                city,
                state,
                country,
                zip_code,
                company_name,
                status,
                address1,
                address2,
                permissions,
                company_logo: req.body.company_logo
            }
        );

        if (result) {
            return response.successResponse(res, 200, result, MESSAGE.CONTRACTOR_UPDATE_SUCCESS);
        } else {
            if (req?.files?.company_logo) {
                mediaUtils.unlink(req.body.company_logo); // remove uploaded compony logo on error
            }
            return response.badRequest(res, 400, MESSAGE.CONTRACTOR_NOT_FOUND);
        }
    } catch (e) {
        if (req?.files?.company_logo) {
            mediaUtils.unlink(req.body.company_logo); // remove uploaded compony logo on error
        }
        console.log(e);
        return response.somethingErrorMsgResponse(res, 500, e);
    }
}


/**
* Get All Contractor from database by role
* @param {string} role - user role
*/
const viewAllContractor = async (req, res) => {
    try {
        let query = await db.checkQueryString(req.query)
        query.role = "CONTRACTOR"
        if (req?.query?.filter && !isNaN(req?.query?.filter))
            query.$or.push({ mobile_no: Number(req.query.filter) })
        return response.successResponse(res, 200, await db.getItems(req, User, query), MESSAGE.CONTRACTOR_VIEW_DATA);
    } catch (error) {
        return response.somethingErrorMsgResponse(res, 500, error);
    }
}


/**
* Get one Contractor from database by id
* @param {string} id - user id
*/
const viewContractor = async (req, res) => {
    const contractor_id = req.query.id;
    try {
        const contractor = await User.findOne({ _id: contractor_id }).exec();
        return response.successResponse(res, 200, contractor);
    } catch (e) {
        return response.somethingErrorMsgResponse(res, 500, e);
    }
}

/**
* Delete one Contractor from database by id
* @param {string} id - user id
*/
const deleteContractor = async (req, res) => {
    try {
        const result = await db.softDeleteItems(req.params.id, User)
        if (result) {
            return response.successResponse(res, 200, {}, MESSAGE.CONTRACTOR_DELETE_SUCCESS);
        } else {
            return response.somethingErrorMsgResponse(res, 500, MESSAGE.SOMETHING_WENT_WRONG);
        }
    } catch (e) {
        return response.somethingErrorMsgResponse(res, 500, e);
    }
}

/**
* toggle change for Contractor status from database by id
* @param {string} id - contractor id
*/
const updateContractorStatus = async (req, res) => {
    const contractor_id = req.query.id;
    User.findOne({ _id: contractor_id }, (err, material) => {
        contractor.status = !contractor.status;
        contractor.save(function (err, updatedContractor) {
            if (!err) {
                return response.successResponse(res, 200, { status: contractor.status }, MESSAGE.CONTRACTOR_STATUS_UPDATED);
            } else {
                return response.somethingErrorMsgResponse(res, 500, error);
            }
        });
    });
}



module.exports = {
    addContractor,
    updateContractor,
    viewContractor,
    deleteContractor,
    viewAllContractor,
    updateContractorStatus,
};
