const response = require("../../constants/response");
const MESSAGE = require("../../constants/message");
const { Building, Media, Project } = require("../../models");
var fs = require("fs");
const db = require("../../services/db");
const mediaUtils = require("../../utils/unlinkMedia");


const add = async (req, res) => {
  try {
    let results = await db.insertItem(Building, req.body);
    let media = [];

    const mediaRootPath = "public/uploads/" + req.body.ownerId + '/buildings/';
    const mediaKey = Object.keys(req.files);
    mediaKey.forEach(async key => {
      if (req.files[key].length && req.files[key].length > 0) {
        // multiple files
        const MediaFiles = req.files[key];
        await MediaFiles.forEach(async mediaFile => {
          media.push({
            path: mediaRootPath + mediaFile.fieldname + '/' + mediaFile.filename,
            filename: mediaFile.filename,
            ref_id: results._id,
            type: mediaFile.fieldname
          });
        })

      } else {
        console.log("Else condition block")
      }
    })
    if (results) {
      // store all images,documents and videos
      await Media.insertMany(media);
      const result = await db.getItemByParams(Building, { _id: results._id }, [{ path: "media" }])

      return response.successResponse(res, 200, result, MESSAGE.BUILDING_ADD_SUCCESS)
    } else {
      // delete all uploaded building media files
      await media.forEach(element => {
        mediaUtils.unlink(element.path);
      });

      return response.badRequest(res, 400, MESSAGE.BUILDING_NOT_FOUND);
    }
  } catch (e) {
    // delete all uploaded building media files
    await deleteMedia(req);
    return response.errorMessageResponse(res, 500, e);
  }
};

/**
 * Gets one Building from database by id
 * @param {string} id - req
 */
const viewOne = async (req, res) => {
  const Building_id = req.query.id;
  try {
    const building = await db.getItemByParams(Building, { _id: Building_id },
      [{ path: "media", match: { deleted: false } }, { path: 'ownerId' }, { path: 'projects' }])
    return response.successResponse(res, 200, building);
  } catch (e) {
    return response.somethingErrorMsgResponse(res, 500, e);
  }
};

/**
 * Gets All Buildings from database accourding to each owner's ID.
 * @param {string} createdBy - req
 */
const viewAll = async (req, res) => {
  try {
    let query = await db.checkQueryString(req.query)
    if (req?.query?.owner && req?.query?.owner_specific)
      query.ownerId = req.query.owner
    // req.populate = [{ path: 'projects', populate: [{ path: 'media' }, { path: 'existingRoof_type' }] }]
    return response.successResponse(res, 200, await db.getItems(req, Building, query), MESSAGE.BUILDING_VIEW_DATA);
  } catch (error) {
    return response.somethingErrorMsgResponse(res, 500, error);
  }
}

/**
 * Gets All Buildings from database.
 * @param {string} createdBy - req
 */
const getAllBuildings = async (req, res) => {
  try {
    let query = await db.checkQueryString(req.query)
    req.populate = [{ path: 'projects', populate: [{ path: 'media' }], match: { deleted: false } }, { path: 'media', match: { deleted: false } }, { path: "ownerId" }]
    return response.successResponse(res, 200, await db.getItems(req, Building, query), MESSAGE.BUILDING_VIEW_DATA);
  } catch (error) {
    return response.somethingErrorMsgResponse(res, 500, error);
  }
}

/**
 * @description Function is For Update building with media.
 * @param {*} req 
 * @param {*} res 
 * @returns 
 */
const update = async (req, res) => {
  try {
    let results = await db.updateItem(req.body._id, Building, req.body)
    let media = [];
    const mediaRootPath = "public/uploads/" + req.body.ownerId + '/buildings/';
    const mediaKey = Object.keys(req.files);
    mediaKey.forEach(async key => {
      if (req.files[key].length && req.files[key].length > 0) {
        // multiple files
        const MediaFiles = req.files[key];
        await MediaFiles.forEach(async mediaFile => {
          media.push({
            path: mediaRootPath + mediaFile.fieldname + '/' + mediaFile.filename,
            filename: mediaFile.filename,
            ref_id: results._id,
            type: mediaFile.fieldname
          });
        })

      } else {
        console.log("Else condition block")
      }
    })
    if (results) {
      await Media.insertMany(media);
      // Remove all the removed media from directory and Media Collection
      if (req.body.removed_items) {
        const filesTobeRemoved = JSON.parse(req.body.removed_items);
        await filesTobeRemoved.forEach(element => {
          mediaUtils.unlink(element.path);
          // delete record from Media collection
          Media.delete({ _id: element._id, filename: element.filename, ref_id: element.ref_id }).exec();
        });
      }
      const result = await db.getItemByParams(Building, { _id: results._id }, [{ path: "media" }]);
      // Update all projects location data belogs to the building
      const project_location = {
        city: req.body.city,
        state: req.body.state,
        country: req.body.country,
        zip_code: req.body.zip_code,
        address: req.body.address1
      }
      let projectUpdateRsults = await Project.updateMany({ buildingId: req.body._id },{ $set: project_location },{ multi: true }).exec();
                  
      return response.successResponse(res, 200, result, MESSAGE.BUILDING_UPDATE_SUCCESS);
    } else {
      // remove all recently uploaded images from directory
      await media.forEach(element => {
        mediaUtils.unlink(element.path);
      });
      return response.badRequest(res, 400, MESSAGE.BUILDING_NOT_FOUND);
    }
  } catch (e) {
    // delete all uploaded building media files
    await deleteMedia(req);
    return response.somethingErrorMsgResponse(res, 500, e);
  }
};

const deleteOne = async (req, res) => {
  const BUILDING_ID = req.body.building_id;
  try {
    await Building.findOneAndUpdate({ _id: BUILDING_ID }, { deleted: true }).exec();
    // Make sure to delete all projects projects as well
    await Project.findOneAndUpdate({ building_id: BUILDING_ID }, { deleted: true }).exec();
    return response.successResponse(res, 200, {}, MESSAGE.BUILDING_DELETE_SUCCESS);
  } catch (e) {
    // come back here
    return response.somethingErrorMsgResponse(res, 500, e);
  }
};

const deleteDocument = async (req, res) => {
  const FILE_ID = req.body.file_id;
  const BUILDING_ID = req.body.building_id;
  try {
    let document = await File.findOneAndRemove({ _id: FILE_ID }).exec();
    fs.unlinkSync(document.path);
    if (document) {
      // remove reference from Building Schema
      Building.updateOne(
        { _id: BUILDING_ID },
        { $pull: { documents: { file_id: FILE_ID } } },
        { safe: true, multi: true },
        function (err, obj) {
          if (!err) {
            return response.successResponse(res, 200, obj, MESSAGE.BUILDING_FILE_DELETE_SUCCESS);
          } else {
            return response.errorMessageResponse(res, 400, MESSAGE.BUILDING_FILE_DELETE_FAIL);
          }
        }
      );
    }
  } catch (e) {
    return response.errorMessageResponse(res, 400, MESSAGE.BUILDING_FILE_DELETE_FAIL);
  }
};

const deleteBuilding = async (req, res) => {
  try {
    const activeProjects = await db.getItemsByParams(Project, { buildingId: req.params.id, status: true }, [], {}, 'project_title');
    if (activeProjects.length > 0) {
      return response.somethingErrorMsgResponse(res, 500, 'Building cannot be deleted. Please close the associated Project(s) first.', activeProjects);
    } else {
      // All Good to remove Building
      const result = await db.softDeleteItems(req.params.id, Building)
      if (result) {
        return response.successResponse(res, 200, {}, MESSAGE.BUILDING_DELETE_SUCCESS);
      } else {
        return response.somethingErrorMsgResponse(res, 500, MESSAGE.SOMETHING_WENT_WRONG);
      }
    }

  } catch (e) {
    return response.somethingErrorMsgResponse(res, 500, e);
  }
}

// Method will delete Media from the directory Path- SS
deleteMedia = async (req) => {
  // delete all uploaded building media files
  const mediaRootPath = "public/uploads/" + req.body.ownerId + '/buildings/';
  const mediaKey = Object.keys(req.files);
  mediaKey.forEach(async key => {
    if (req.files[key].length && req.files[key].length > 0) {
      // multiple files
      const MediaFiles = req.files[key];
      await MediaFiles.forEach(async mediaFile => {
        // call util method to remove file from location
        mediaUtils.unlink(mediaRootPath + mediaFile.fieldname + '/' + mediaFile.filename);
      })
    }
  })
}

module.exports = {
  add,
  viewOne,
  viewAll,
  update,
  deleteOne,
  deleteDocument,
  getAllBuildings,
  deleteBuilding
};
