const response = require("../../constants/response");
const MESSAGE = require("../../constants/message");
const { Bid, Project,Media, Invitation, User } = require("../../models");
const mail = require("../../utils/notify");
const { consoleSandbox } = require("@sentry/utils");
const db = require("../../services/db");
/**
 * Returns the list of the bids made by contractors on the projects
 * @param {object} req.body
 * @param {*}  
 * @returns 
 */
const allBidByProjectId = async (req, res) => {
  let req_Data = req.body;

  let { limit, page, project_Id, ...sortby } = req_Data;

  var sortName = sortby.sort ? sortby.sort : "createdAt";
  var sortOption = {};
  sortOption[`${sortName}`] =
    sortby.order && sortby.order == 1 ? sortby.order : -1;

  try {
    const all_bids = await Bid.paginate(
      { project_Id: project_Id },
      { page: page || 1, limit: limit || 10, sort: sortOption, populate: { path: 'contractor_id' } }
    )
    if (all_bids) {
      return response.successResponse(
        res,
        200,
        all_bids,
        MESSAGE.ALL_BID_SUCCESS
      );
    } else {
      return response.badRequest(res, 400, MESSAGE.BID_NOT_FOUND);
    }
  } catch (error) {
    return response.somethingErrorMsgResponse(res, 500, error);
  }
};
/**
 * Returns the all the details of the bid
 * @param {string} id 
 * @param {*}  
 * @returns 
 */

const viewBid = async (req, res) => {
  const bid_id = req.query.id;
  try {
    const bid = await Bid.findOne({ _id: bid_id }).exec();
    return response.successResponse(res, 200, bid);
  } catch (e) {
    return response.somethingErrorMsgResponse(res, 500, e);
  }
};


/**
 * Changes the status of the project
 * @param {boolean} status 
 * @param {string} project_Id
 * @returns 
 */

const changeStatus = async (req, res) => {
  const { status, project_Id } = req.body.status;
  try {
    var result = await Project.findByIdAndUpdate((_id = project_Id), {
      status: status,
    });
    if (result) {
      return response.successResponse(
        res,
        200,
        result,
        MESSAGE.BID_UPDATE_SUCCESS
      );
    } else {
      return response.badRequest(res, 400, MESSAGE.BID_NOT_FOUND);
    }
  } catch (e) {
    return response.somethingErrorMsgResponse(res, 500, e);
  }
};

/**
 * Changes the bid period
 * @param {Date} bid_start_date 
 * @param {Date} bid_end_date 
 * @param {string} project_Id 
 * @returns 
 */

const changeBidPeriod = async (req, res) => {
  const { bid_start_date, bid_end_date, project_Id } = req.body;
  try {
    var result = await Project.findByIdAndUpdate((_id = project_Id), {
      bid_start_date: bid_start_date,
      bid_end_date: bid_end_date,
    });
    if (result) {
      return response.successResponse(
        res,
        200,
        result,
        MESSAGE.BID_UPDATE_SUCCESS
      );
    } else {
      return response.badRequest(res, 400, MESSAGE.BID_NOT_FOUND);
    }
  } catch (e) {
    return response.somethingErrorMsgResponse(res, 500, e);
  }
};

/**
 * Add contractors to the bid
 * @param {string} project_Id 
 * @param {string} contractor_id 
 * @returns 
 */
const addContrator = async (req, res) => {
  const { project_Id, contractor_id } = req.body;
  try {
    const bid = new Bid({
      project_Id,
      contractor_id,
      isDirectEntry: true,
    });
    const saveBid = await bid.save();
    return response.successResponse(res, 200, saveBid, MESSAGE.BID_ADD_SUCCESS);
  } catch (e) {
    return response.somethingErrorMsgResponse(res, 500, e);
  }
};

/**
 * Send bid invitation to the contractos
 * @param {Object} req 
 * @param {*} res 
 * @returns 
 */
const sendInvitationToContrator = async (req, res) => {
  try {
    const {email, bid_start, bid_end, timezone, project_Id} = req.body;
    const invitation = new Invitation({
      project_Id: project_Id,
      contarctor_email: email,
      bid_start: bid_start,
      bid_end: bid_end,
      timezone: timezone,
      createdBy: req.id,
    });
    await invitation.save();

    let subject = "Bid Invitation";
    let text = `hi inviatation for bid`;
    mail.sendMail(email, subject, text);
    return response.successResponse(res, 200, {}, MESSAGE.BID_INVITATION_SUCCESS);
  } catch (e) {
    return response.somethingErrorMsgResponse(res, 500, e);
  }
};

/**
 * Make the contractor winner of the bid
 * @param {string} bidId 
 * @param {string} projectId 
 * @returns 
 */
const makeBidWinner = async (req, res) => {
  const { bidId, projectId } = req.body;
  try {
    const bids = await Bid.find({
      project_Id: projectId,
      is_bid_winner: true
    }).exec()
    if (Array.isArray(bids) && bids.length > 0) {
      return response.conflictErrorMsgResponse(res, 409, MESSAGE.BID_WINNER_ALREADY_EXIST);
    }
    const result = await Bid.findByIdAndUpdate((_id = bidId), { is_bid_winner: true }, { returnOriginal: false });
    if (result) {
      return response.successResponse(res, 200, result, MESSAGE.BID_UPDATE_SUCCESS);
    } else {
      return response.badRequest(res, 400, MESSAGE.BID_NOT_FOUND);
    }
  } catch (e) {
    return response.somethingErrorMsgResponse(res, 500, e);
  }

}

// API for bidden project for web
const biddenprojects = async (req, res) => {
   try {
        const obj = { contractor_id: req.query.id };
        const result = await Bid.find(obj);
        let arr = [];
        result.forEach(element => {
            arr.push(element.project_Id);
        });
        const biddenProjects = await Project.find({ '_id': { $in: arr } })
            .populate({ path: 'media' });
        if(biddenProjects.length <= 0)
        return response.successResponse(res, 200,{}, MESSAGE.BIDDEN_PROJECT_NOT_FOUND);
        else 
          return response.successResponse(res, 200, biddenProjects, MESSAGE.BIDDEN_PROJECT_VIEW_DATA);

    } catch (error) {
        return response.somethingErrorMsgResponse(res, 500, error);
    }
}


/**
 * @description Function For List of all Bidden with respective projects.
 * @param {object} req 
 * @param {object} res 
 */
 const viewAllBids = async (req, res) => {
  try {
    let query = await db.checkQueryString(req.query)
    req.populate = [{ path: 'project_Id' ,populate: { path: 'media' } }]
    return response.successResponse(res, 200, await db.getItems(req, Bid, query), MESSAGE.OWNER_VIEW_DATA);
  } catch (error) {
    return response.somethingErrorMsgResponse(res, 500, error);
  }
}
// API to get list of Bidders for a perticular project
const getBiddersByProjectId = async (req, res) => {
  try {
    const bidders = await Bid.find({project_Id:req.body.projectId}).populate({path:'contractor_id'});
    let arr = [];
    bidders.forEach(element => {
          arr.push(element.contractor_id.email);
       });
      return response.successResponse(res, 200, bidders, MESSAGE.BIDDER_LIST_OF_PROJECT);
  } catch (error) {
      return response.somethingErrorMsgResponse(res, 500, error);
    }
}
// API for sending Email to single or multiple users
// pass array like---> "email":["testmail@gmail.com","tastemail1@gmail.com"]
const sendEmailtoBidders = async (req, res) => {
  try {
    let email = req.body.email;
    let subject = "Greetings for project Bid";
    let text = `Dear Bidder,<br>This info is about project Bid`;
    mail.sendMail(email, subject, text);
    return response.successResponse(res, 200,email,MESSAGE.BID_EMAIL_SENT_TO_BIDDER);
  } catch (e) {
    return response.somethingErrorMsgResponse(res, 500, e);
  }
}
module.exports = {
  allBidByProjectId,
  viewBid,
  changeStatus,
  changeBidPeriod,
  addContrator,
  sendInvitationToContrator,
  makeBidWinner,
  biddenprojects,
  viewAllBids,
  sendEmailtoBidders,
  getBiddersByProjectId,
};
