/*
    This is authentication controller for all types of user authentication 
*/
require("dotenv").config();
// const jwt = require("jsonwebtoken");
const date = require("date-and-time");
const jwt = require("jsonwebtoken");
const bcrypt = require("bcryptjs");
const stripe = require('stripe')(process.env.STRIPE_SECURITY_KEY);
const saltRounds = 10;
const mail = require("../utils/notify");
const verify = require("../utils/verifyEmailAndOtp");
const { getJwtToken } = require("../utils/jwtToken");
const { _number } = require("../utils/getRandom");
const { User, Session, ContactUs, Media, UserPermission, Module, Project } = require("../models/");
const response = require("../constants/response");
const MESSAGE = require("../constants/message");
const { emailStatus } = require("../utils/getUser");
const db = require("../services/db");
const { _value } = require("../utils/getRandom");
const now = new Date();
const mediaUtils = require("../utils/unlinkMedia");
const EMAIL_TEMPLATES = require("../constants/email_templates_id");


const login = async (req, res) => {
  const { email, password } = req.body;
  // get user along with assign modules and same orders
  let user = await db.getItemByParams(User,{ email });

  if (!user || user.deleted)
    return response.somethingErrorMsgResponse(res, 200, MESSAGE.USER_NOT_FOUND);

  let passwordIsValid = bcrypt.compareSync(password, user.password);
  if (passwordIsValid) {
    // check if active
    if (!user.status)
      return response.somethingErrorMsgResponse(res, 403, MESSAGE.NOT_ACTIVE);

    // START: map assign modules in order which aranged while assign permissions from front-end request

    //this need to be udated with populate
    let accesses = [];
    if (user.role === "ADMIN" || user.role === "RPC_MANAGER") {
      const permissions = await UserPermission.findOne({ user_id: user._id }).exec();
      console.log("permissions", permissions)
      accesses = Array.isArray(permissions.module) ? [...permissions.module] : []
    }
    // if (permissions && permissions.module) {
    // modules = await Module.find({
    //   '_id': {
    //     $in: permissions.module
    //   }
    // }).exec();



    // user.modules = modules
    // }
    // Create payload for jwt
    // const token = getJwtToken({
    //   _id: user._id,
    //   _role_name: user.role,
    // });
    const token = jwt.sign(
      {
        _id: user._id,
        _role_name: user.role,
      },
      process.env.TOKEN_SECRET,
      {
        expiresIn: '24h',
      }
    );
    return response.successResponse(res, 200, {
      token, user,
      modules: accesses
    }, MESSAGE.LOGIN_SUCCESS);
  } else {
    return response.somethingErrorMsgResponse(res, 200, MESSAGE.LOGIN_FAILED);
  }
};

const signup = async (req, res) => {
  const { first_name, last_name, email, role, company_name, mobile_no, address1, address2, city, state, country, zip_code, phone_no, toll_free_no } = req.body;
  // For Company Logo
  if (req?.files?.company_logo) {
    req.body.company_logo = "public/uploads/" + req.files.company_logo[0].filename;
  }
  let MailExist = await emailStatus(email);
  // if mail exist then return message
  if (MailExist) {
    mediaUtils.unlink(req.body.company_logo); // remove uploaded compony logo on error
    return response.somethingErrorMsgResponse(res, 403, MESSAGE.EMAIL_ALREADY_REGISTERED)
  }
  try {
    const secquenceCount = await User.find({}).count() + 1;
    const temp_password = _value(10);
    // Random Generate Password for User that is get by on user's email......
    const password = bcrypt.hashSync(temp_password, 16);

    let user = new User({
      first_name,
      last_name,
      email,
      password: password,
      role,
      modifiedAt: date.format(now, "YYYY/MM/DD HH:mm:ss"),
      token: null,
      company_name,
      mobile_no,
      address1,
      address2,
      city,
      state,
      country,
      zip_code,
      toll_free_no,
      phone_no,
      user_id: req.body.role + '_' + secquenceCount,
      company_logo: req.body.company_logo
    });
    user = await user.save();
    const payload = {
      _id: user._id,
      _role_name: role,
    };
    const jwtToken = getJwtToken(payload);
    resObject = {
      jwtToken,
      expires_in: process.env.EXPIRES_IN || 86000,
    };
    // let subject = "Your Password ";
    // let text = `hi ${email} <br> and your password is ${temp_password}`;
    // mail.sendMail(email, subject, text);

    const emailData = {
      msg: {
        to: email,
        from: 'Winston@roofingprojects.com', // Use the email address or domain you verified above
        template_id: EMAIL_TEMPLATES.USER_ONBOARDING_EMAIL_TEMPLATE_ID,
        dynamic_template_data:{
          "first_name": first_name,
          "last_name": last_name,
          "password": temp_password
        }
      }
    };
    mail.sendMailWithTemplate(emailData);

    await db.insertItem(UserPermission, {
      user_id: user._id,
      module: req.body.module
    })

    return response.successResponse(res, 200, resObject, MESSAGE.SIGNUP_SUCCESS);
  } catch (error) {
    mediaUtils.unlink(req.body.company_logo); // remove uploaded compony logo on error
    return response.somethingErrorMsgResponse(res, 301, error);
  }
};

// forget password API for Website
const forgetPassword = async (req, res) => {
  const { email } = req.body;
  try {
    const userDetail = await User.findOne({ email });
    if (userDetail) {
      let payload = { _id: userDetail._id, _email: email };
      let verify = getJwtToken(payload);
      const filter = { _id: userDetail._id };
      const update = { token: verify };
      let result = await User.findOneAndUpdate(filter, update, { new: true });
      // let subject = "Link for change password";
      // let text = `hi ${email} <br> <a href = "${process.env.FRONTEND_RESET_PASSWORD_URL}/?assertionToken=${result.token}&email=${email}">Click here</a> to reset your password`;
      // mail.sendMail(email, subject, text);
      const emailData = {
        msg: {
          to: email,
          from: 'Winston@roofingprojects.com', // Use the email address or domain you verified above
          template_id: EMAIL_TEMPLATES.FORGOT_PASSWORD_LINK_TEMPLATE_ID,
          dynamic_template_data:{
            "first_name": userDetail.first_name,
            "last_name": userDetail.last_name,
            "reset_link": `"${process.env.FRONTEND_RESET_PASSWORD_URL}/?assertionToken=${result.token}&email=${email}"`
          }
        }
      };
      mail.sendMailWithTemplate(emailData);
      return response.successResponse(res, 200, {}, MESSAGE.RESET_LINK_SUCCESS);
    }
    else
      return response.somethingErrorMsgResponse(res, 500, MESSAGE.EMAIL_NOT_FOUND);
  } catch (error) {
    console.log(error)
    return response.somethingErrorMsgResponse(res, 500, error, MESSAGE.SOMETHING_WENT_WRONG);
  }
};
// API for Email verification
const verifyToken = async (req, res) => {
  const { email, token } = req.body;
  const result = await User.findOne({ email: email });
  if (result.token != token) {
    return response.somethingErrorMsgResponse(res, 403, MESSAGE.TOKEN_ERROR);
  }
  jwt.verify(token, process.env.TOKEN_SECRET, (err, decoded) => {
    if (err)
      return response.somethingErrorMsgResponse(res, 403, MESSAGE.TOKEN_ERROR);
    else return response.successResponse(res, 200, MESSAGE.TOKEN_VALIDATED);
  });
};
// Reset Password API for Website
const resetPassword = async (req, res) => {
  try {
    const { password, token } = req.body;
    let decode = jwt.verify(token, process.env.TOKEN_SECRET);
    let result = await User.find({ email: decode._email });
    if (decode._email) {
      if (result) {
        if (result[0].token == null)
          response.somethingErrorMsgResponse(res, 301, MESSAGE.TOKEN_IS_NULL);
        else {
          bcrypt.genSalt(saltRounds, function (err, salt) {
            bcrypt.hash(password, salt, function (err, hash) {
              verify.updateWithObject(
                User,
                { email: result[0].email },
                { password: hash, token: null },
              );
              verify.updateWithObject(User, { email: result[0].email }, { password: hash, token: null });
            });
          });
          response.successResponse(res, 200, {}, MESSAGE.PASSWORD_UPDATE_SUCCESS);
        }
      }
      else
        response.somethingErrorMsgResponse(res, 301, MESSAGE.PASSWORD_UPDATE_FAILURE);
    }
  } catch (error) {
    response.somethingErrorMsgResponse(res, 301, MESSAGE.SOMETHING_WENT_WRONG);
  }
};
/**
 * Gets user data from database by id
 */
const getUserById = async (req, res) => {
  const { id } = req;
  try {
    const userData = await User.findById(id).exec();
    return response.successResponse(res, 200, userData, MESSAGE.GET_USER_DATA);
  } catch (error) {
    return response.somethingErrorMsgResponse(res, 500, error);
  }
};

const logout = async (req, res) => {
  try {
    const { id } = req;
    await Session.deleteMany({ user_id: id });
    return response.successResponse(res, 200, {}, MESSAGE.LOGOUT_SUCCESS);
  } catch (e) {
    return response.somethingErrorMsgResponse(res, 500, error);
  }
};

const contactUs = async (req, res) => {
  try {
    const { first_name, last_name, email, phone, subject, query } = req.body;
    const contact = new ContactUs({
      first_name,
      last_name,
      phone,
      email,
      subject,
      query,
    });
    await contact.save();
    // let data = "RoofingProjects";
    // let text = `hi ${first_name} <br> Your contact enquiry has been forwarded, we will contact you soon`;
    // mail.sendMail(email, data, text);
    const emailData = {
      msg: {
        to: email,
        from: 'Winston@roofingprojects.com', // Use the email address or domain you verified above
        template_id: EMAIL_TEMPLATES.CUSTOMER_SUPPORT_TEMPLATE_ID,
        dynamic_template_data:{
          "first_name": first_name,
          "last_name": last_name,
          "query_subject":subject,
          "customer_message":query,
        }
      }
    };
    mail.sendMailWithTemplate(emailData);
    return response.successResponse(res, 200, {}, MESSAGE.CONTACT_US_SUCCESS);
  } catch (e) {
    return response.somethingErrorMsgResponse(res, 500, error);
  }
};


const login_new = async (req, res) => {
  const { email, password } = req.body;

  const user = await User.findOne({ email: email });
  
  if (user && (await bcrypt.compare(password, user.password)))  {
    // check if active
    if (!user.status){
      return response.somethingErrorMsgResponse(res, 403, MESSAGE.NOT_ACTIVE);
    }
    const accesses = [] ;
    const token = jwt.sign(
      {_id: user._id, _role_name: user.role},
      process.env.TOKEN_SECRET,
      {
        expiresIn: '24h',
      }
    );

    return res.status(200).json({data:{modules: accesses,token: token ,user: user}, message: "Login success"});

  } else {
    // return response.somethingErrorMsgResponse(res, 200, MESSAGE.LOGIN_FAILED);
    return res.status(400).json({ "is_error": true, "message": "Invalid Credentials." });

  }
};

module.exports = {
  login,
  signup,
  forgetPassword,
  resetPassword,
  getUserById,
  verifyToken,
  logout,
  contactUs,
  login_new
};
